// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.ipfix

import scala.collection.mutable.HashMap
import scala.collection.immutable.HashSet
import java.io.ObjectInputStream

import com.typesafe.scalalogging.{LazyLogging, StrictLogging}

/**
  * The InfoModel class represents the IPFIX Information Model.
  *
  * The Information Model contains [[InfoElement Information Elmenets]].
  *
  * To use the standard IPFIX Information Model, the caller invokes
  * the `getStandardInfoModel()` method on the [[InfoModel$ companion
  * object]].  If necessary, that function reads the standard model
  * from an XML file.  The caller is given an empty model that
  * inherits from the standard model.  This allows the caller to
  * manipulate the information model without invalidating the standard
  * model or disrupting other threads.
  *
  * @param parent The model that this model inherits from
  *
  * @see [[InfoModel$ The companion object]] for more details.
  */
final class InfoModel private (val parent: Option[InfoModel] = None)
    extends Serializable with LazyLogging
{

  /**
    * A map from Identifier (elementId) to InfoElement
    */
  private final var idmap: HashMap[Identifier, InfoElement] =
    new HashMap[Identifier, InfoElement]()

  /**
    * A map from Name to InfoElement
    */
  @transient private var namemap: HashMap[String, InfoElement] =
    new HashMap[String, InfoElement]()

  /**
    * A map from enterpriseId to the function to call to generate a
    * reverse element within that enterprise.
    */
  @transient private var reversemap:
      HashMap[Long, InfoModel.MakeReversibleElement] =
    new HashMap[Long, InfoModel.MakeReversibleElement]()

  //@transient private[this] ReentrantReadWriteLock rwlock;
  //@transient private[this] Lock readlock;
  //@transient private[this] Lock writelock;
  //private[this] static final ReentrantLock standard_lock = new ReentrantLock();

  private def makeIterator(): Iterator[InfoElement] = {
    // readlock.lock()
    try {
      namemap.values.iterator
    } finally {
      // readlock.unlock()
    }
  }


  /**
    * Creates an [[InfoElement]] from each node in `nodes` and adds
    * that InfoElement to the InfoModel.
    *
    * @param nodes The sequence of Nodes to parse
    */
  def importElements(nodes: scala.xml.NodeSeq): Unit = {
    for (n <- nodes) {
      try {
        val ie = InfoElement.fromXML(n)
        add(ie)
      } catch {
        case e @ (_ : InvalidInfoElementException
           | _ : IllegalInfoElementAttributeException)
            => logger.debug(
              "Invalid information element record: " + e.getMessage())
      }
    }
  }


  /**
    * Determines whether reverse elements are included with the
    * standard model.  Returns `true` if previous setting was to
    * include reverse elements, and `false` if the previous setting
    * was not to include reverse elements.
    */
  def setCreateStandardReverse(withReverse: Boolean): Boolean = {
    val prev =
      if (withReverse) {
        reversemap.put(0L, InfoModel.reverseStandard)
      } else {
        reversemap.remove(0L)
      }
    prev.nonEmpty
  }


  /**
    * Registers the class or object `mre` as being the one to use when
    * attempting to find the corresponding reverse [[InfoElement]] for
    * an `InfoElement` whose enterpriseId is `pen`.
    */
  def addReverseMapper(pen: Long, mre: InfoModel.MakeReversibleElement):
      Option[InfoModel.MakeReversibleElement] =
    reversemap.put(pen, mre)

  /**
    * Registers an object that creates a reverse [[InfoElement]] by
    * masking the elementId of the element's [[Identifier]] by `bits`
    * and prepending "reverse" to the element's name when the
    * enterpriseId of the element is `pen`.
    */
  def addReverseMapper(pen: Long, bits: Int):
      Option[InfoModel.MakeReversibleElement] =
    addReverseMapper(pen, new InfoModel.SimpleReversePrivateElement(bits))

  /**
    * Registers an object that creates a reverse [[InfoElement]] by
    * masking the elementId of the element's [[Identifier]] by 0x4000
    * and prepending "reverse" to the element's name when the
    * enterpriseId of the element is `pen`.
    */
  def addReverseMapper(pen: Long): Option[InfoModel.MakeReversibleElement] =
    addReverseMapper(pen, InfoModel.REVERSE_IE_PRIVATE_BITS)


  private[this] def addInfoElement(ie: InfoElement): Unit = {
    var old = idmap.put(ie.ident, ie)
    for (e <- old) {
      namemap.remove(e.name)
    }
    old = namemap.put(ie.name, ie)
    for (e <- old) {
      idmap.remove(e.ident)
    }
  }

  /**
    * Adds the information element `ie` and the corresponding reverse
    * element if there is a known function to do that mapping.
    */
  def add(ie: InfoElement): Unit = {
    //writelock.lock()
    try {
      addInfoElement(ie)
      reversemap.get(ie.enterpriseId).foreach {
        mre => mre.reverse(ie).foreach {
          rie => addInfoElement(rie)
        }
      }
    } finally {
      //writelock.unlock()
    }
  }


  /**
    * Gets the information element from the model that has the specified
    * `elementId` and `enterpriseId'.
    *
    * @param elemendId The element ID of the element to find.
    * @param enterpriseId The Private Enterprise Number of the element to find.
    * @throws InvalidInfoElementException if an InfoElement is not found
    * @since 1.3.1
    */
  def apply(elementId: Int, enterpriseId: Long = 0L): InfoElement = {
    val id = try {
      Identifier(elementId, enterpriseId)
    } catch {
      case e: IllegalInfoElementAttributeException =>
        throw new InvalidInfoElementException(
          s"Invalid Identifier: ${e.toString}")
    }
    apply(id)
  }

  /**
    * Gets the information element from the model that has the specified
    * Identifier.
    * @param id Identifier of the element to retrieve
    * @throws InvalidInfoElementException if an InfoElement is not found
    * @since 1.3.1
    */
  def apply(id: Identifier): InfoElement =
    get(id).getOrElse(throw new InvalidInfoElementException(
      s"No InfoElement in the InfoModel has Identifier ${id.toString}"))

  /**
    * Gets the information element from the model that has the specified
    * `name`.
    * @param name Name of the element to retrieve
    * @throws InvalidInfoElementException if an InfoElement is not found
    * @since 1.3.1
    */
  def apply(name: String): InfoElement =
    get(name).getOrElse(throw new InvalidInfoElementException(
      s"No InfoElement in the InfoModel has name '${name}'"))

  /**
    * Returns `true` if this model contains an [[InfoElement]] that has the
    * specified `elementId` and `enterpriseId'; returns `false` otherwise.
    * Also returns `false` if `elementId` or `enterpriseId` are outside the
    * range of legal values.
    *
    * @param elemendId The element ID of the element to find.
    * @param enterpriseId The Private Enterprise Number of the element to find.
    */
  def contains(elementId: Int, enterpriseId: Long = 0L): Boolean = {
    try {
      contains(Identifier(elementId, enterpriseId))
    } catch {
      case _: IllegalInfoElementAttributeException => false
    }
  }

  /**
    * Returns `true` if this model contains an [[InfoElement]] that has the
    * specified Identifier.
    */
  def contains(id: Identifier): Boolean = get(id).nonEmpty

  /**
    * Returns `true` if this model contains an [[InfoElement]] that has the
    * specified `name`.  Returns `false` otherwise.
    */
  def contains(name: String): Boolean = get(name).nonEmpty

  /**
    * Gets the information element from the model that has the
    * specified `elementId` and `enterpriseId` as an [[scala.Option Option]].
    * Returns [[None]] if `elementId` or `enterpriseId` are outside the
    * range of legal values.
    *
    * @param elemendId The element ID of the element to find.
    * @param enterpriseId The Private Enterprise Number of the element to find.
    */
  def get(elementId: Int, enterpriseId: Long = 0L): Option[InfoElement] = {
    //logger.trace(s"Checking model for elements whose element id is ${elementId} and enterpriseId is ${enterpriseId}")
    try {
      get(Identifier(elementId, enterpriseId))
    } catch {
      case _: IllegalInfoElementAttributeException => None
    }
  }

  /**
    * Gets the information element from the model that has the
    * specified identifier as an [[scala.Option Option]].
    */
  def get(id: Identifier): Option[InfoElement] = {
    idmap.get(id) match {
      case Some(elem) =>
        if (elem != InfoModel.deletedElement) {
          Option(elem)
        } else {
          None
        }
      case None =>
        parent flatMap { _.get(id) }
    }
  }

  /**
    * Gets the information element from the model that has the
    * specified `name` as an [[scala.Option Option]].
    */
  def get(name: String): Option[InfoElement] = {
    namemap.get(name) match {
      case Some(elem) =>
        if (elem != InfoModel.deletedElement) {
          Option(elem)
        } else {
          None
        }
      case None =>
        parent flatMap { _.get(name) }
    }
  }

  /**
    * Marks as deleted the information element that has the specified
    * identifier.  Returns the deleted element.
    */
  def remove(id: Identifier): Option[InfoElement] = {
    // writelock.lock()
    try {
      idmap.remove(id) match {
        case Some(elem) =>
          if (elem != InfoModel.deletedElement) {
            // since local to this map, remove it from namemap
            namemap.remove(elem.name)
            return Option(elem)
          } else {
            // do not allow removal of deletedElement
            idmap.put(id, InfoModel.deletedElement)
          }
        case None =>
          if (parent.nonEmpty) {
            parent.get.get(id) match {
              case Some(elem) =>
                if (elem != InfoModel.deletedElement) {
                  idmap.put(id, InfoModel.deletedElement)
                  namemap.put(elem.name, InfoModel.deletedElement)
                  return Option(elem)
                }
              case None =>
            }
          }
      }
    } finally {
      // writelock.unlock()
    }
    None
  }

  /**
    * Marks as deleted the information element that has the specified
    * name.  Returns the deleted element.
    */
  def remove(name: String): Option[InfoElement] = {
    // writelock.lock()
    try {
      namemap.remove(name) match {
        case Some(elem) =>
          if (elem != InfoModel.deletedElement) {
            // since local to this map, remove it from idmap
            idmap.remove(elem.ident)
            return Option(elem)
          } else {
            // do not allow removal of deletedElement
            namemap.put(name, InfoModel.deletedElement)
          }
        case None =>
          if (parent.nonEmpty) {
            parent.get.get(name) match {
              case Some(elem) =>
                if (elem != InfoModel.deletedElement) {
                  namemap.put(name, InfoModel.deletedElement)
                  idmap.put(elem.ident, InfoModel.deletedElement)
                  return Option(elem)
                }
              case None =>
            }
          }
      }
    } finally {
      // writelock.unlock()
    }
    None
  }

  //override def clone(): AnyRef = InfoModel(this)

  /**
    * Returns an [[scala.collection.Iterator Iterator]] to visit the
    * [[InfoElement InfoElements]] in the model.
    */
  def iterator: Iterator[InfoElement] =
    new InfoModel.IEIterator(this)


  private[this] def readObject(in: ObjectInputStream): Unit = {
    in.defaultReadObject()
    // create locks
    reversemap = new HashMap[Long, InfoModel.MakeReversibleElement]()
    namemap = new HashMap[String, InfoElement]() ++= (
      for (ie <- idmap.values) yield (ie.name, ie)
    )
  }

}


/**
  * An [[InfoModel]] factory.
  */
object InfoModel extends StrictLogging {
  private final val REVERSE_IE_PRIVATE_BITS: Int = 0x4000

  private final val CERT_PEN: Long = 6871

  /**
    * Classes or objects that extend this trait define a `reverse()`
    * method that maps an [[InfoElement]] to its corresponding reverse
    * information element.
    */
  trait MakeReversibleElement {
    /**
      * Returns the corresponding reverse information element for the
      * argument or returns `None` when there is no reverse element.
      */
    def reverse(ie: InfoElement): Option[InfoElement]
  }


  /**
    * When an element is removed from a model where the element is
    * defined in the parent (or grandparent) of this model, this
    * element is inserted into the model to explicitly mark the
    * element as deleted.
    */
  private val deletedElement = InfoElement.fromXML(
    <record>
      <enterpriseId>{CERT_PEN}</enterpriseId>
      <elementId>{0x7fff}</elementId>
      <name>deletedElement</name>
      <description>A deleted element</description>
      </record>)

  /**
    * This is an object to map a standard information to its reverse
    * element.
    */
  private object ReverseStandardIE extends MakeReversibleElement {
    private final val REVERSE_IE_PEN: Long = 29305L /* rfc 5103 */
    private val nonReversableElements: HashSet[Integer] = HashSet(
      /* From RFC 5103 6.1 1 */
      148,            /* flowId */
      145,            /* templateId */
      149,            /* observationDomainId */
      137,            /* commonPropertiesId */
      /* From RFC 5103 6.1 4 */
      210,            /* paddingOctets */
      /* From RFC 5103 6.1 5 */
      239             /* biflowDirection */
    )
    private val nonReversableGroups: HashSet[String] = HashSet(
      /* From RFC 5103 6.1 2 */
      "config",
      /* From RFC 5103 6.1 3 */
      "processCounter"
    )

    override def reverse(ie: InfoElement): Option[InfoElement] = {
      assert(ie.enterpriseId == 0)
      if (nonReversableElements.contains(ie.elementId)
        || nonReversableGroups.contains(ie.group))
      {
        return None
      }
      val b = InfoElementBuilder(ie)
      val name = ie.name
      try {
        b.name = ("reverse" + name.substring(0, 1).toUpperCase()
          + name.substring(1))
        b.enterpriseId = REVERSE_IE_PEN
        Option(b.build())
      } catch {
        case e @ (_ : IllegalInfoElementAttributeException
           | _ : InvalidInfoElementException) => throw new RuntimeException(e)
      }
    }
  }

  private final class SimpleReversePrivateElement(bits: Int)
      extends MakeReversibleElement
  {
    override def reverse(ie: InfoElement): Option[InfoElement] = {
      assert(0 != ie.enterpriseId)
      val id: Int = ie.elementId
      if ((id & bits) == bits) {
        // already a reverse element
        return None
      }
      val b: InfoElementBuilder = InfoElementBuilder(ie)
      val name: String = ie.name
      try {
        b.name = ("reverse" + name.substring(0, 1).toUpperCase()
          + name.substring(1))
        b.elementId = (id | bits)
        Option(b.build())
      } catch {
        case e @ (_ : IllegalInfoElementAttributeException
           | _ : InvalidInfoElementException) => throw new RuntimeException(e)
      }
    }
  }


  private final class IEIterator(model: InfoModel)
      extends Iterator[InfoElement]
  {
    private[this] var it: Iterator[InfoElement] = model.makeIterator()
    private[this] var next: Option[InfoElement] = None
    private[this] var leaf_iter: Boolean = true

    private[this] def getNext(): Boolean = {
      if (!it.hasNext) {
        if (!leaf_iter || model.parent.isEmpty) {
          // no more levels to climb
          return false
        }
        it = model.parent.get.iterator
        leaf_iter = false
        return getNext()
      }
      var e = it.next()
      if (!leaf_iter) {
        var b = model.idmap.contains(e.ident)
        while (b && it.hasNext) {
          e = it.next()
          b = model.idmap.contains(e.ident)
        }
        if (b) {
          return false
        }
      }
      next = Option(e)
      true
    }

    override def hasNext: Boolean = {
      next.nonEmpty || getNext()
    }

    override def next(): InfoElement = {
      if (next.isEmpty && !getNext()) {
        throw new NoSuchElementException
      }
      val ie = next.get
      next = None
      ie
    }
  }


  private val reverseStandard: MakeReversibleElement =
    ReverseStandardIE
  //private final val standardLock: ReentrantLock = new ReentrantLock()

  private var standardModel: Option[InfoModel] = None

  private var standardReverse: Option[InfoModel] = None


  /**
    * Returns a new IPFIX Information Model that inherits from the
    * standard Information Model.
    *
    * If `includeReverse` is `true`, the new model includes reverse
    * elements.
    *
    * If there is no current info model, the model is loaded from
    * disk.
    *
    * @return An InfoModel inheriting from the standard InfoModel
    */
  def getStandardInfoModel(
    includeReverse: Boolean = true,
    reparse:        Boolean = false):
      InfoModel =
  {
    //standardLock.lock()
    try {
      if (standardModel.isEmpty) {
        // parse the XML
        val m = new InfoModel()
        try {
          val stream = InfoModelRegistry("ipfix")
          logger.debug("Reading standard info model from resources...")
          val elem = xml.XML.load(stream)
          //LOG//logger.trace("element label is " + elem.label)
          //LOG//logger.trace("element text is " + elem.text.substring(0, 40))
          if (elem.label != "registry") {
            throw new RuntimeException("did not find expected label")
          }
          val nodes = (elem \ "registry")
          //LOG//logger.trace("nodes length is " + nodes.size)
          //LOG//logger.trace("nodes text is " + nodes.text.substring(0, 40))
          val node = nodes(0)
          //LOG//logger.trace("node label is " + node.label)
          //LOG//logger.trace("node text is " + node.text.substring(0, 40))
          //LOG//logger.trace("node id is " + node.label)
          if ((node \ "@id").text != "ipfix-information-elements") {
            throw new RuntimeException("did not find expected tag")
          }
          m.importElements(node \ "record")
          logger.debug("Finished reading standard info model.")
          standardModel = Option(m)
        } catch {
          case e: Any => throw e
        }

        // create a version of the standard model that includes
        // reverse elements
        val rev = new InfoModel()
        rev.setCreateStandardReverse(true)
        for (ie <- standardModel.get.iterator) {
          rev.add(ie)
        }
        standardReverse = Option(rev)
      }
      // return the requested model
      if (includeReverse) {
        inheritInfoModel(standardReverse.get)
      } else {
        inheritInfoModel(standardModel.get)
      }
    } finally {
      //standardLock.unlock()
    }
  }

  def getCERTStandardInfoModel(): InfoModel = {
    try {
      val m: InfoModel = InfoModel(getStandardInfoModel())
      m.addReverseMapper(CERT_PEN)
      val stream = InfoModelRegistry("cert_ipfix")
      logger.debug("Reading CERT info model from resources...")
      val elem = xml.XML.load(stream)
      if (elem.label != "registry") {
        throw new RuntimeException("did not find expected label")
      }
      val node = (elem \ "registry")(0)
      if ((node \ "@id").text != "cert-information-elements") {
        throw new RuntimeException("did not find expected tag")
      }
      m.importElements(node \ "record")
      logger.debug("Finished reading CERT info model.")
      m
    } catch {
      case e: Any => throw e
    }
  }


  /**
    * Creates a complete empty information model that inherits from
    * nothing.  This is defined for use during testing.
    */
  def empty(): InfoModel = {
    new InfoModel(None)
  }


  /**
    * Creates a new information model that inherits from `parent`.
    *
    * @param parent The model to inherit from.
    */
  def inheritInfoModel(parent: InfoModel): InfoModel = {
    new InfoModel(Option(parent))
  }


  /**
    * Creates a new copy of an InfoModel
    *
    * @param source The InfoModel to create a copy of
    */
  def apply(source: InfoModel): InfoModel = {
    val m = new InfoModel(source.parent)
    try {
      //source._readlock.lock()
      m.idmap = for ((k, v) <- source.idmap) yield (k -> v)
      m.namemap = for ((k, v) <- source.namemap) yield (k -> v)
      m.reversemap = for ((k, v) <- source.reversemap) yield (k -> v)
      m
    } finally {
      //source._readlock.unlock()
    }
  }

}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
