// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.ipfix

import java.io.InputStream
import java.nio.channels.Channels
import java.nio.channels.FileChannel
import java.nio.channels.ReadableByteChannel
import java.nio.file.Path
import java.nio.file.StandardOpenOption
import java.util.concurrent.Callable


/**
  * An InputStreamSession takes a readable channel, a file path, or
  * an input stream, creates a [[StreamMessageReader]] to read
  * [[Message Messages]] from the stream, creates a [[RecordReader]]
  * to parse the messages into [[Record Records]], and wraps the
  * RecordReader's `call()` method.
  *
  * @param channel The file channel to read
  * @param model The information model to use
  */
final class InputStreamSession private (
  channel: ReadableByteChannel,
  model: InfoModel)
    extends Callable[Option[Record]]
{
  /** The SessionGroup for the [[Session Sessions]] created while
    * reading the stream. */
  val sessionGroup = SessionGroup(model, channel)

  private[this] val reader: RecordReader =
    RecordReader(StreamMessageReader(channel, sessionGroup))

  /**
    * Registers `callback` with the [[SessionGroup]] used by the
    * [[RecordReader]].
    */
  def register(callback: Session.TemplateCallback): Unit = {
    sessionGroup.register(callback)
  }

  override def call(): Option[Record] =
    if ( reader.hasNext ) {
      Option(reader.next())
    } else {
      None
    }
}

/**
  * A [[InputStreamSession]] factory.
  */
object InputStreamSession {
  /**
    * A [[InputStreamSession]] factory.
    *
    * @param channel The channel to read the [[Record Records]] from
    * @param model The information model to use
    */
  def apply(channel: ReadableByteChannel, model: InfoModel): InputStreamSession=
    new InputStreamSession(channel, model)

  /**
    * A [[InputStreamSession]] factory.
    *
    * @param path The file path to open for reading the [[Record Records]]
    * @param model The information model to use
    */
  def apply(path: Path, model: InfoModel): InputStreamSession =
    new InputStreamSession(FileChannel.open(path, StandardOpenOption.READ),
      model)

  /**
    * A [[InputStreamSession]] factory.
    *
    * @param stream The input stream to read the [[Record Records]] from
    * @param model The information model to use
    */
  def apply(stream: InputStream, model: InfoModel): InputStreamSession =
    new InputStreamSession(Channels.newChannel(stream), model)
}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
