// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.ipfix

import java.nio.{BufferOverflowException, BufferUnderflowException, ByteBuffer}
import scala.collection.mutable.ArrayBuffer


/**
  * The IpfixSet class represents an IPFIX Set.  (The name is "IpfixSet" to
  * avoid conflicts with the standard Scala [[scala.collection.Set Set]]).

  * A Set is a generic term for a collection of records that have a
  * similar structure.  A Set consists of a 4-byte Set Header and one
  * or more records.
  *
  * There are three types of Sets: (1)Data Sets contain IPFIX flow
  * records and are represented by the [[RecordSet]] class.
  * (2)Template Sets and (3)Option Template Sets contain templates (or
  * schemas) that describe the representation of the data in Data
  * Sets.  They are represented by the [[TemplateSet]] class.
  *
  * To create a Set from IPFIX data stored in a [[java.nio.ByteBuffer]]
  * that was read as part of [[Message message]], use
  * {{{
  * val set = IpfixSet.fromBuffer(buffer, message)
  * }}}
  *
  * @param id The identifier for this Set, which is a [[Template]]
  * identifier if the Set is [[RecordSet]].
  * @param session The [[Session]] which the Set was read from or will
  * be written to.
  *
  * @see [[IpfixSet$ The companion object]] for more details
  * @since 1.3.1 This class was previously called `Set`.
  */
sealed abstract class IpfixSet protected (
  final val id: Int,
  final val session: Session)
{
  /**
    * Returns the number of items (Data Records or Template Records)
    * in the Set.
    */
  def size: Int

  /**
    * Returns the number of octets required to write this Set to an
    * IPFIX stream.
    */
  //def octetLength: Int

  /**
    * Appends the Set to a buffer for writing to an IPFIX stream.
    *
    * @throws java.nio.BufferOverflowException if there is not enough
    * room in the buffer for all items in the Set.  The buffer's
    * position when an error is thrown unknown.
    */
  def toBuffer(buffer: ByteBuffer): ByteBuffer

  /**
    * Returns an Iterator over the Templates in this IpfixSet, returning an
    * empty Iterator if this is a RecordSet.
    */
  def templateIterator: Iterator[Template] = {
    this match {
      case ts: TemplateSet => ts.iterator
      case _: RecordSet => Iterator.empty
    }
  }

  /**
    * Returns an Iterator over the Records in this IpfixSet, returning an
    * empty Iterator if this is a TemplateSet.
    */
  def recordIterator: Iterator[Record] = {
    this match {
      case _: TemplateSet => Iterator.empty
      case rs: RecordSet => rs.iterator
    }
  }

  /**
    * Returns a String representation the Set.
    */
  override def toString(): String = s"Set(Id: $id, ItemCount: $size)"
}


/**
  * An [[IpfixSet]] factory.
  */
object IpfixSet {
  /**
    * The octet length of a Set header.
    */
  val headerLength: Int = 4

  /**
    * Processes the data in the ByteBuffer to create either a
    * [[TemplateSet]] or a [[RecordSet]].  When this function returns,
    * `buffer` is positioned immediately after the data that was read
    * to create the Set.
    *
    * @param buffer The ByteBuffer containing the data representing a Set.
    * @param message The IPFIX Message that contains this Set.
    */
  def fromBuffer(buffer: ByteBuffer, message: Message): IpfixSet = {
    val (settype: Int, length: Int) = try {
      (0xffff & buffer.getShort(buffer.position()),
        0xffff & buffer.getShort(buffer.position() + 2))
    } catch {
      case _: BufferUnderflowException =>
        throw new TruncatedReadException(
          "Not enough octets available for a Set header" +
            s" (${buffer.remaining()} of ${headerLength})")
    }

    if (length < headerLength) {
      throw new TruncatedReadException(
        s"Set header contains an illegal length (${length})")
    }
    if (length > buffer.remaining()) {
      throw new TruncatedReadException(
        s"Length in Set header (${length}) is larger than octets available" +
          s" (${buffer.remaining()})")
    }
    // create a slice for this Set
    val setbuffer: ByteBuffer = buffer.slice()
    setbuffer.limit(length)
    // move buffer to start of next Set
    buffer.position(buffer.position() + length)
    // create the approprate set
    if (settype >= MIN_TEMPLATE_ID) {
      RecordSet.fromBuffer(settype, setbuffer, message)
    } else if (settype==TEMPLATE_SET_ID || settype==OPTIONS_TEMPLATE_SET_ID) {
      TemplateSet.fromBuffer(settype, setbuffer, message)
    } else {
      throw new IllegalSetException(s"ID in Set header is invalid (${settype})")
    }
  }
}



/**
  * A TemplateSet represents a [[IpfixSet]] that contains one or more
  * [[Template Templates]] or Option Templates.
  *
  * To create a TemplateSet from an input ByteBuffer, call
  * [[Set$ IpfixSet]].[[IpfixSet#fromBuffer fromBuffer()]].
  *
  * To create a TemplateSet for export, use [[TemplateSet$
  * TemplateSet].[[TemplateSet#empty() empty()]] to create an empty
  * TemplateSet and then call addTemplate() to add Templates to it.
  *
  * @see [[TemplateSet$ The companion object]] for more details.
  */
final class TemplateSet private (
  id: Int,
  session: Session,
  private[this] var templates: Array[Template],
  private[this] var exportLength: Int = IpfixSet.headerLength)
    extends IpfixSet(id, session) with Iterable[Template]
{
  /* NOTE: This is a val that is set creation time---The Set is
   * read-only if the set of templates are known at creation time. */
  private[this] val readOnly: Boolean = (templates.length > 0)

  /**
    * Indicates whether this template set holds option templates.
    */
  val isOptionsTemplate: Boolean = (OPTIONS_TEMPLATE_SET_ID == id)

  /**
    * Returns an Iterator over the Templates in the Set.
    */
  def iterator: Iterator[Template] = templates.iterator

  /**
    * Returns the number of Templates in the Set.
    */
  override def size: Int = templates.size

  /**
    * Returns the total octetLength of this TemplateSet when reading
    * from or writing to a buffer.
    */
  def octetLength: Int = exportLength

  /**
    * Appends the TemplateSet to a buffer for writing to an IPFIX
    * stream.
    *
    * @throws java.nio.BufferOverflowException if there is not enough
    * room in the buffer for all Templates in the TemplateSet.
    * Because the method makes multiple writes to the buffer, the
    * buffer's position is unknown if an error is thrown.
    */
  def toBuffer(outbuf: ByteBuffer): ByteBuffer = {
    assert(templates.size > 0)
    val pos = outbuf.position()
    outbuf.putShort(id.toShort)
      .putShort(0.toShort)
    for (t <- templates) {
      t.toBuffer(outbuf, session)
    }
    outbuf.putShort(pos + 2, (outbuf.position() - pos).toShort)
  }


  /**
    * Adds a Template to the TemplateSet.  The Option setting of the
    * Template must match that of the Template Set.
    *
    * @throws RuntimeException when either the Template Set is
    * read-only or the isOptionsTemplate setting of the Template does
    * not match that of the Template Set.
    * @throws MessageTooLongException when adding the Template would
    * cause the size of the exported Template Set to exceed the maximum
    * IPFIX message size.
    */
  def addTemplate(tmpl: Template): TemplateSet = {
    if (readOnly) {
      // FIXME
      throw new RuntimeException("TemplateSet is read-only")
    }
    if (isOptionsTemplate != tmpl.isOptionsTemplate) {
      val (typeT, typeTS) =
        if (isOptionsTemplate) { ("a non-", "an ") } else { ("an ", "a non-") }
      throw new RuntimeException(
        s"Cannot add ${typeT}options template to ${typeTS}options template set")
    }
    val prevLen = exportLength
    exportLength += tmpl.octetLength
    //println(s"TemplateSet: exportLength = $exportLength," +
    //  s" prev = $prevLen, this = ${exportLength - prevLen}")
    if (exportLength > EXPORT_BUFFER_USABLE_LEN) {
      exportLength = prevLen
      throw new MessageTooLongException("Unable to add template")
    }
    templates = templates :+ tmpl
    this
  }

}

/**
  * A [[TemplateSet]] factory.
  */
object TemplateSet {
  /**
    * Processes the data in the ByteBuffer to create a [[TemplateSet]].
    * Expects `buffer` to contain only this Template Set.
    *
    * This is intended to be called by IpfixSet.fromBuffer().
    *
    * @param id The Set ID specified in the buffer.
    * @param buffer The ByteBuffer containing the data representing a set.
    * @param message The IPFIX message that contains this set.
    */
  private[ipfix] def fromBuffer(id: Int, buffer: ByteBuffer, message: Message):
      TemplateSet =
  {
    val isOption: Boolean = (OPTIONS_TEMPLATE_SET_ID == id)
    val minSize = if (isOption) { 6 } else { 4 }

    val builder = ArrayBuffer.empty[Template]
    buffer.position(IpfixSet.headerLength)
    while ( buffer.remaining() >= minSize ) {
      builder += Template.fromBuffer(buffer, message, isOption)
    }

    new TemplateSet(id, message.session, builder.toArray, buffer.limit())
  }

  /**
    * Creates an empty [[TemplateSet]] to hold [[Template Templates]]
    * for writing to `session`.  The optional `isOptionSet` parameter
    * determines whether the TemplateSet holds Option Templates.
    */
  def empty(session: Session, isOptionSet: Boolean = false): TemplateSet = {
    new TemplateSet(
      if (isOptionSet) { OPTIONS_TEMPLATE_SET_ID } else { TEMPLATE_SET_ID },
      session, Array.empty[Template])
  }
}



/**
  * A RecordSet represents an [[IpfixSet]] that contains one or more
  * Data [[Record Records]].
  *
  * To create a RecordSet from an input ByteBuffer, call
  * [[IpfixSet$ IpfixSet]].[[IpfixSet#fromBuffer fromBuffer()]].
  *
  * To create a RecordSet for export, use [[RecordSet$
  * RecordSet]].[[RecordSet#empty empty()]] to create an empty
  * RecordSet and then call addRecord() to add Records to it.
  *
  * @see [[RecordSet$ The companion object]] for more details.
  */
final class RecordSet private (
  id: Int,
  session: Session,
  final val template: Template,
  private[this] var records: Array[Record],
  private[this] var exportLength: Int = IpfixSet.headerLength)
    extends IpfixSet(id, session) with Iterable[Record]
{
  private[this] var exportBuffer: ByteBuffer = _

  /* NOTE: This is a val that is set creation time---The Set is
   * read-only if the set of records are known at creation time. */
  private[this] val readOnly: Boolean = (records.length > 0)

  /**
    * Returns an Iterator over the Records in the Set.
    */
  def iterator: Iterator[Record] = records.iterator

  /**
    * Returns the number of Records in the Set.
    */
  override def size: Int = records.size

  /**
    * Appends the RecordSet to a buffer for writing to an IPFIX
    * stream.
    *
    * Assumes all [[Template Templates]] used by the [[Record
    * Records]] in this Set have been written to the [[Session]].
    *
    * @throws java.nio.BufferOverflowException if there is not enough
    * room in the buffer for all Records in the RecordSet.
    * Because the method may make multiple writes to the buffer, the
    * buffer's position is unknown if an error is thrown.
    */
  def toBuffer(outbuf: ByteBuffer): ByteBuffer = {
    assert(records.size > 0)
    assert(exportBuffer.position() > IpfixSet.headerLength)
    val pos = exportBuffer.position()
    try {
      exportBuffer.putShort(0, id.toShort)
      exportBuffer.putShort(2, pos.toShort)
      exportBuffer.flip()
      outbuf.put(exportBuffer)
    } catch {
      // re-position the internal exportBuffer on error
      case e: BufferOverflowException =>
        exportBuffer.position(pos)
        throw e
    }
  }

  /**
    * Adds a Record to the RecordSet.
    *
    * @throws RuntimeException when either the ID of the Record's
    * [[Template]] does not match the ID specified in the Record Set
    * or the Record's Template's [[Session]] must match the Session of
    * the Record Set.
    * @throws MessageTooLongException when adding the record would
    * cause the size of the exported Record Set to exceed the maximum
    * IPFIX message size.
    */
  def addRecord(rec: Record): RecordSet = {
    if (readOnly) {
      throw new RuntimeException("RecordSet is read-only")
    }
    if (template != rec.template) {
      // FIXME
      throw new RuntimeException(
        "Template of RecordSet does not match that of the Record")
    }
    if (Option(exportBuffer).isEmpty) {
      exportBuffer = ByteBuffer.allocate(EXPORT_BUFFER_USABLE_LEN)
      exportBuffer.position(IpfixSet.headerLength)
    }
    val pos = exportBuffer.position()
    try {
      rec.toBuffer(exportBuffer, session)
      //println(s"RecordSet(${template.id}): exportLength" +
      //  s" = ${exportBuffer.position()}; prev = $pos;" +
      //  s" this = ${exportBuffer.position() - pos};" +
      //  s" octetLength = ${rec.octetLength}")
    } catch {
      case (_ : BufferOverflowException | _ : IllegalArgumentException) =>
        exportBuffer.position(pos)
        throw new MessageTooLongException("Unable to add record")
    }
    exportLength = exportBuffer.limit()
    records = records :+ rec
    this
  }

}

/**
  * A [[RecordSet]] factory.
  */
object RecordSet {
  /**
    * Processes the data in the ByteBuffer to create a [[RecordSet]].
    * Expects `buffer` to contain only this Record Set.
    *
    * This is intended to be called by IpfixSet.fromBuffer().
    *
    * If the RecordSet contains template-metadata records, that metadata is
    * added to the [[Session]] associated with `message`.
    *
    * If the RecordSet contains element-definition records (RFC 5610), those
    * elements are added to the [[InfoModel]] associated with the message's
    * [[Session]].
    *
    * @param id The Set ID specified in the buffer.
    * @param buffer The ByteBuffer containing the data representing a Set.
    * @param message The IPFIX message that contains this set.
    * @throws NoTemplateException if there is no [[Template]] for `id`
    */
  private[ipfix] def fromBuffer(id: Int, buffer: ByteBuffer, message: Message):
      RecordSet =
  {
    val template = message.session(id)
    val minSize = template.minimumLength

    val builder = ArrayBuffer.empty[Record]
    buffer.position(IpfixSet.headerLength)
    if ( !template.isMetadataTemplate ) {
      while ( buffer.remaining() >= minSize ) {
        builder += template.readRecord(buffer, message.session, Option(message))
      }
    } else if ( template.isTemplateMetadataTemplate ) {
      while ( buffer.remaining() >= minSize ) {
        val r = template.readRecord(buffer, message.session, Option(message))
        builder += r
        val metadata = TemplateMetadata.createFrom(r)
        if ( metadata.name.nonEmpty ) {
          message.session.addTemplateMetadata(metadata)
        }
      }
    } else {
      assert(template.isInfoElementMetadataTemplate)
      val model = message.session.infoModel
      while ( buffer.remaining() >= minSize ) {
        val r = template.readRecord(buffer, message.session, Option(message))
        builder += r
        val metadata = InfoElementMetadata.createFrom(r)
        if ( !model.contains(metadata.id, metadata.pen) ) {
          model.add(InfoElement.fromInfoElementMetadata(metadata))
        }
      }
    }
    new RecordSet(
      id, message.session, template, builder.toArray, buffer.limit())
  }

  /**
    * Creates an empty [[RecordSet]] to hold [[Record Records]] whose
    * [[Template]] is `template` and that will be written to `session`.
    *
    * Adds `template` to the `session` if needed (and assigns an arbitrary
    * ID).
    */
  def empty(template: Template, session: Session): RecordSet = {
    val id = session.getOrAdd(template)
    new RecordSet(id, session, template, Array.empty[Record])
  }
}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
