// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.ipfix

import java.nio.ByteBuffer
import scala.collection.immutable.{Set => ScalaSet}


/**
  * An abstract class that represents an element in a [[Record]] that is a
  * structured data; i.e., a [[BasicList]], a [[SubTemplateList]], or
  * a [[SubTemplateMultiList]].
  */
abstract class ListElement() {
  /** Read the semanticId */
  protected def readSemanticId(buffer: ByteBuffer): Short = {
    (buffer.get() & 0xff).toShort
  }

  /** The ID of the semantic value for the list. */
  protected val semanticId: Short
  // abstract value

  /**
    * Returns the sematics of the elements in this list.
    */
  final lazy val semantics: ListSemantics = ListSemantics.withValue(semanticId)

  /**
    * Returns a Record that represents an entry in the list.
    * @param idx The index of the entry to return.  Entries are
    * numbered from 0 up to `size`.
    */
  def apply(idx: Int): Record
  // abstract method

  /**
    * Returns the number of entries in the list.
    */
  def size: Int
  // abstract method

  /**
    * Returns an iterator over the records in the list.
    */
  def iterator: Iterator[Record]
  // abstract method

  /**
    * Appends the list to a buffer for writing to an IPFIX stream.
    */
  def toBuffer(outbuf: ByteBuffer, session: Session): ByteBuffer
  // abstract method

  /**
    * Returns the length of the ListElement when writing to an IPFIX
    * stream.
    */
  def octetLength: Int
  // abstract method

  /**
    * Creates a Set containing the `Template`s used by the list and any
    * sub-lists, recursively.
    */
  def allTemplates: ScalaSet[Template]
  // abstract method

  /**
    * Creates a Set containing the [[InfoElement]] used by any [[BasicList]]
    * on this list and its sub-lists, recursively.
    */
  def allBasicListElements: ScalaSet[InfoElement]
  // abstract method

  def toString(): String
  // abstract method

  def formatted: String

}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
