// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.ipfix

import enumeratum.values.{ShortEnum, ShortEnumEntry}

/**
  * IPFIX Structured Data (List) Type Semantics as defined in RFC6313
  *
  * @see [[ListSemantics$ The companion object]] for a complete
  * description and the list of values.
  *
  * @param value The short value representing the list semantics
  * @param name The name of the list semantics.
  */
sealed abstract class ListSemantics(val value: Short, val name: String)
    extends ShortEnumEntry

/**
  * IPFIX Structured Data (List) Type Semantics as defined in RFC6313
  *
  * Structured data type semantics are provided in order to express
  * the relationship among multiple list elements in a Structured Data
  * Information Element.
  *
  * See https://www.iana.org/assignments/ipfix/ipfix.xhtml#ipfix-structured-data-types-semantics
  *
  * May be converted to and from [[scala.Short Short]] values.
  */
case object ListSemantics extends ShortEnum[ListSemantics] {
  /**
    * Collection of all known valid list semantics.
    * @group Members
    */
  lazy val values = findValues
  // Note: lazy to avoid cyclic initialization problem

  /**
    * Converts a short value to a list semantic type.
    * @return The list semantic represented by this value.
    * @throws java.util.NoSuchElementException if the short represents
    *     no known list semantic.
    * @group Members
    */
  def apply(v: Short): ListSemantics =
    withValueOpt(v).getOrElse(
      throw new NoSuchElementException(
        s"Unrecognized list semantic value $v")
    )

  /**
    * Finds the IPFIX list semantic type whose name is `name`.
    *
    * The function seaches linearly through the list of list semantics.
    *
    * @return The list semantic represented by `name`.
    * @throws java.util.NoSuchElementException if the name represents
    *     no known list semantic.
    */
  def withName(name: String): ListSemantics =
    values.find(_.name == name).getOrElse(
      throw new NoSuchElementException(
        s"Unrecognized information element semantic $name")
    )

  /**
    * Finds the IPFIX IE list semantic value whose name is `name` as
    * an Option.
    *
    * The function seaches linearly through the list of list semantics.
    *
    * @return The list semantic represented by `name`.
    */
  def withNameOpt(name: String): Option[ListSemantics] =
    values.find(_.name == name)


  /**
    * IPFIX List semantic denoting the semantic is not specified.
    * @group Values
    */
  case object Undefined extends ListSemantics(0xff, "undefined")
  /**
    * IPFIX list semantic value denoting that none of the elements are
    * properties of the data record.
    * @group Values
    */
  case object NoneOf extends ListSemantics(0, "noneOf")
  /**
    * IPFIX list semantic value denoting that only a single element in
    * the list is a property of the data record.
    * @group Values
    */
  case object ExactlyOneOf extends ListSemantics(1, "exactlyOneOf")
  /**
    * IPFIX list semantic value denoting one or more elements in the
    * list are properties of the data record.
    * @group Values
    */
  case object OneOrMoreOf extends ListSemantics(2, "oneOrMoreOf")
  /**
    * IPFIX list semantic value denoting all of the elements in the
    * list are properties of the data record.
    * @group Values
    */
  case object AllOf extends ListSemantics(3, "allOf")
  /**
    * IPFIX list semantic value denoting elements in the list are
    * ordered.
    * @group Values
    */
  case object Ordered extends ListSemantics(4, "ordered")
}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
