// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.ipfix

import com.github.ghik.silencer.silent
import scala.collection.immutable.{Set => ScalaSet}
import java.nio.ByteBuffer
import java.time.Instant

import java.lang.reflect.{Field => MemberField}
import java.util.WeakHashMap
import java.util.concurrent.ConcurrentHashMap


/**
  *  A Record instance represents an IPFIX data record.  Each record
  *  contains a set of values which are described by the record's
  *  [[Template]].
  *
  *  Each IPFIX data type can be mapped to its Java data
  *  type as in the table below:
  *
  *   - IPFIX data type ⟶ Scala data type
  *   - `octetArray` ⟶ Seq[Byte]`<sup>1</sup>
  *   - `unsigned8` ⟶ `Int`
  *   - `unsigned16` ⟶ `Int`
  *   - `unsigned32` ⟶ `Long`
  *   - `unsigned64` ⟶ `Long`<sup>2</sup>
  *   - `signed8` ⟶ `Int`
  *   - `signed16` ⟶ `Int`
  *   - `signed32` ⟶ `Int`
  *   - `signed64` ⟶ `Long`
  *   - `float32` ⟶ `Float`
  *   - `float64` ⟶ `Double`
  *   - `boolean` ⟶ `Boolean`
  *   - `macAddress` ⟶ `Seq[Byte]`<sup>1,3</sup>
  *   - `string` ⟶ `String`
  *   - `dateTimeSeconds` ⟶ `[[java.time.Instant Instant]]`
  *   - `dateTimeMilliseconds` ⟶ `[[java.time.Instant Instant]]`
  *   - `dateTimeMicroseconds` ⟶ `[[java.time.Instant Instant]]`
  *   - `dateTimeNanoseconds` ⟶ `[[java.time.Instant Instant]]`
  *   - `ipv4Address` ⟶ `[[org.cert.netsa.data.net.IPv4Address IPv4Address]]`
  *   - `ipv6Address` ⟶ `[[org.cert.netsa.data.net.IPv6Address IPv6Address]]`
  *   - `basicList` ⟶ `[[BasicList]]`
  *   - `subTemplateList` ⟶ `[[SubTemplateList]]`
  *   - `subTemplateMultiList` ⟶ `[[SubTemplateMultiList]]`
  *
  * <sup>1</sup> The `Seq[Byte]` returned for `octetArray` or
  * `macAddress` is a read-only view on the underlying record data. If
  * you want to save the data without keeping the full record data
  * from being garbage collected, you should copy the `Seq`.
  *
  * <sup>2</sup> `Long` is signed, so an `unsigned64` with its high
  * bit set will appear as negative.
  *
  * <sup>3</sup> The `Seq[Byte]` returned for a `macAddress` will
  * always be of length 6.
  *
  * A record's fields are described by a [[Template]], which may be
  * retrieved from the record using the [[#template]] value.
  * Using the data from the template, the user can cast the record's
  * objects to the appropriate Java type.
  *
  * Alternatively, a record's values can be retrieved in a typed
  * manner using either a [[FieldExtractor]], or a [[Fillable]]
  * object.  A FieldExtractor acts as a reference to a particular
  * field in a record regardless of the current template of the
  * record.  A Fillable object is an object which contains fields that
  * have been marked with the [[IPFIXExtract]] annotation.  A record's
  * [[#fill fill()]] method can be used to "fill in" the object's
  * fields from the record.
  *
  * @param template The Template that describes this Record.
  *
  * @see [[FieldExtractor]]
  * @see [[Fillable]]
  * @see [[Template]]
  */
abstract class Record protected (final val template: Template) {
  /**
    * Gets the value within the record for the given field by field
    * position within the [[Template]].
    *
    * @param idx The index of the field whose value to return.
    * @return The value for the referenced field.
    * @throws java.lang.IndexOutOfBoundsException if the index is out
    * of range
    * {{{(idx < 0 || idx >= template.size())}}}
    */
  def apply(idx: Int): Any

  /**
    * Gets the value within the record for the field referenced by the
    * extractor as an Option.
    *
    * @tparam T The result type of the extractor.
    * @param extractor A field extractor.
    * @return The value for the referenced field, or [[scala.None]] if
    *         the extractor does not match a field in this record.
    */
  final def apply[T](extractor: FieldExtractor[T]): Option[T] =
    extractor.extractFrom(this)

  /**
    * Gets the value within the record for the given field by finding the
    * field matching a FieldSpec within the [[Template]].
    *
    * @param spec The field to find with the record's Template in order to get
    * its position.
    * @return The value for the referenced field.
    * @throws java.lang.IndexOutOfBoundsException if the Template does not
    * contain the FieldSpec.
    * @since 1.3.1
    */
  def apply(spec: FieldSpec): Any = apply(template.indexOf(spec))

  /**
    * Gets the value within the record for the given field by finding the
    * field matching an [[InfoElement]] within the [[Template]].
    *
    * @param ie The InfoElement to find with the record's Template in order to
    * get its position.
    * @return The value for the referenced field.
    * @throws java.lang.IndexOutOfBoundsException if the Template does not
    * contain the InfoElement.
    * @since 1.3.1
    */
  def apply(ie: InfoElement): Any = apply(template.indexOf(ie))

  /**
    * Gets the raw byte sequence representing the value for the
    * given field position.  This byte sequence is in network byte
    * order.
    *
    * @param idx The position of the field whose value to return.
    * @return The raw value of the referenced field.
    * @throws java.lang.IndexOutOfBoundsException if the position is
    *         out of range {@code (idx < 0 || idx > template.size())}
    */
  //---UNUSED---
  //def getRaw(idx: Int): List[Byte]

  /**
    * Fills the fields of `obj` that are marked with the
    * [[IPFIXExtract]] annotation from this record.
    *
    * @param obj  the object to fill
    * @see IPFIXExtract
    */
  final def fill(obj: Fillable): Unit = {
    Record.fillHelper(obj, this)
  }

  /**
    * Gets the number of fields in the record.
    */
  final lazy val size: Int = template.size

  /**
    * Gets the export time of the [[Message]] that generated this
    * record as an Option.
    *
    * @return The export time of the Message or [[scala.None None]] if
    * the record represents a list element or was not read from a
    * Message.
    */
  def exportTime: Option[Instant]

  /**
    * Returns the observation domain of the [[Message]] that generated
    * this record as an Option or [[scala.None None]] if the record
    * represents a list element or was not read from a Message.
    */
  def observationDomain: Option[Int]

  /**
    * Returns the Message object from which the Record was read as an
    * Option or [[scala.None]] if the Record was not read from a
    * Message.
    */
  def message: Option[Message]

  /**
    * Modifies the current record so that it carries as little data as
    * necessary to exist by itself.  More specifically, it detaches
    * itself from as much of its parent [[Message Message's]] data as
    * possible.  If non-complete subsets of `Record`s are going
    * to be kept in memory, they should be detached so they don't
    * carry the memory of any discarded `Record`s by association
    * to their parent `Message`s.
    *
    * @return the record itself (not a new record)
    */
  def detach(): Record

  /** Returns an Iterator where iteration yields a Field. */
  final def fields: Iterator[Field] = {
    val t = template
    for (idx <- Iterator.range(0, t.size))
    yield Field(t(idx), apply(idx), t.elementLength(idx), idx)
  }

  /**
    * Appends this Record to a buffer for writing to an IPFIX stream.
    * The function uses the template IDs in `session` if the Record
    * contains SubTemplateLists or a SubTemplateMultiList.  Assumes
    * the [[Template Template(s)]] used by the Record have already
    * been added to the Session and appeneded to the buffer.
    */
  def toBuffer(outbuf: ByteBuffer, session: Session): ByteBuffer = {
    // CollectedRecord overrides this method for conditions when
    // the contents may be copied without decoding
    for (i <- 0 until template.size) {
      val ie = template(i)
      val len = template.elementLength(i)
      val obj = apply(i)
      ie.dataType.toBuffer(outbuf, session, len, obj)
    }
    outbuf
  }

  /**
    * Gets a Set containing the `Template` used by the record and the
    * `Template`s used by any [[ListElement ListElements]] in the
    * record.
    */
  final def allTemplates(): ScalaSet[Template] = {
    var s = ScalaSet[Template](template)
    for ( i <- 0 until template.size ) {
      template(i).dataTypeId match {
        case DataTypes.BasicList =>
          s = s ++ apply(i).asInstanceOf[BasicList].allTemplates
        case DataTypes.SubTemplateList =>
          s = s ++ apply(i).asInstanceOf[SubTemplateList].allTemplates
        case DataTypes.SubTemplateMultiList =>
          s = s ++ apply(i).asInstanceOf[SubTemplateMultiList].allTemplates
        case _ =>
      }
    }
    s
  }

  /**
    * Gets a Set containing the `InfoElement` used by any basic list on the
    * record and by the record's [[ListElement ListElements]].
    */
  final def allBasicListElements(): ScalaSet[InfoElement] = {
    var s = ScalaSet.empty[InfoElement]
    for ( i <- 0 until template.size ) {
      template(i).dataTypeId match {
        case DataTypes.BasicList =>
          s = s ++ apply(i).asInstanceOf[BasicList].allBasicListElements
        case DataTypes.SubTemplateList =>
          s = s ++ apply(i).asInstanceOf[SubTemplateList].allBasicListElements
        case DataTypes.SubTemplateMultiList =>
          s = s ++ apply(i).asInstanceOf[SubTemplateMultiList].allBasicListElements
        case _ =>
      }
    }
    s
  }

  /**
    * Gets the number of octets required to write the record to a
    * stream.
    */
  def octetLength: Int

  override def toString(): String = {
    val sb = new StringBuilder()
    (for (f <- fields) yield s"${f.ie.name}=${f.value}").
      addString(sb, "Record(", ", ", ")")
    sb.mkString
  }

  def formatted: String = {
    import org.apache.commons.text.StringEscapeUtils.escapeJson
    def formatValue(v: Any): String = v match {
      case le: ListElement => le.formatted
      case lv: util.ListView => lv.formatted
      case s: String => s"""\"${escapeJson(s)}\""""
      case _ => v.toString()
    }
    fields.map(f => s"${f.ie.name}: ${formatValue(f.value)}")
      .mkString("{", ", ", "}")
  }

}


/**
  * A [[Record]] copier.
  */
object Record {

  /**
    * Creates a new Record (specifically an [[ArrayRecord]]) as a copy
    * of `record`.
    */
  def apply(record: Record, deep: Boolean = false): Record = {
    val tmpl = record.template
    val newRec = ArrayRecord(tmpl)
    for (i <- 0 until tmpl.size) {
      newRec.update(i, (tmpl(i).dataTypeId match {
        case DataTypes.BasicList =>
          BasicList(record.apply(i).asInstanceOf[BasicList], deep)
        case DataTypes.SubTemplateList =>
          SubTemplateList(record.apply(i).asInstanceOf[SubTemplateList], deep)
        case DataTypes.SubTemplateMultiList =>
          SubTemplateMultiList(
            record.apply(i).asInstanceOf[SubTemplateMultiList], deep)
        case _ =>
          record.apply(i)
      }))
    }
    newRec
  }


  /*
   * *****  Support for the Fillable class  *****
   */

  /**
    * For the class of every object that is passed to the Record's
    * fill() method, an entry is added to this Map from the class to a
    * FieldMappings instance (defined next). */
  private final val fillables =
    new ConcurrentHashMap[Class[_ <: Fillable], FieldMappings]()

  /**
    * A FieldMapping is created for each unique Class and it is
    * comprised of two equally-sized Vectors.  One vector contains the
    * set of @IPFIXExtract annotated fields that exist on the class;
    * the other vector contains the [[FieldSpec]] objects that are
    * created by the annotation's parameters.
    *
    * Each time a FieldMapping is used by a Record, an entry is added
    * to the `map` member linking the Record's Template to a Vector of
    * `Option[Int]`s that specify the index of each FieldSpec in the
    * Template.
    */
  private final class FieldMappings(
    final val fields: Vector[MemberField],
    final val fieldspecs: Vector[FieldSpec])
  {
    final val map = new WeakHashMap[Template, Vector[Option[Int]]]()
  }

  /**
    * Use `rec` to fill `obj`.
    *
    * If the Class of `obj` has not been seen before, find each of
    * its @IPFIXExtract annotations and create a FieldSpec for each.
    * Use those values to create a FieldMappings instance and store it
    * in the `fillables` Map.
    *
    * If the Template of `rec` has not been seen before, find the
    * index of each FieldSpec in the Template and store that result in
    * the `map` member of FieldMappings.
    *
    * Finally, for each index, copy that value from the Record and set
    * the corresponding member of the Fillable object.
    */
  private def fillHelper(obj: Fillable, rec: Record): Unit = {
    val tmpl: Template = rec.template
    val cls: Class[_ <: Fillable] = obj.getClass
    // get or create the cached data for this class and this template
    val (mapping, indices) =
      Option(fillables.get(cls)) match {
        case Some(m) => {
          // we have the mapping; check for indices
          Option(m.map.get(tmpl)) match {
            case Some(index) => (m, index)
            case None =>
              val index = Vector.empty[Option[Int]] ++ (
                for (spec <- m.fieldspecs) yield spec.findIn(tmpl)
              )
              m.map.put(tmpl, index)
              (m, index)
            }
        }
        case None =>
          var fields = Vector.empty[MemberField]
          var fieldspecs = Vector.empty[FieldSpec]
          var index = Vector.empty[Option[Int]]
          for (field <- cls.getDeclaredFields) {
            if (field.isAnnotationPresent(classOf[IPFIXExtract])) {
              if (!field.isAccessible(): @silent) {
                field.setAccessible(true)
              }
              val annot: IPFIXExtract =
                field.getAnnotation(classOf[IPFIXExtract])
              val spec = FieldSpec(annot.name, annot.nth)
              fields = fields :+ field
              fieldspecs = fieldspecs :+ spec
              index = index :+ spec.findIn(tmpl)
            }
          }
          val m = new FieldMappings(fields, fieldspecs)
          m.map.put(tmpl, index)
          fillables.put(cls, m)
          (m, index)
      }
    // set values on this object
    for (i <- 0 until indices.length) {
      try {
        indices.apply(i).foreach { x => {
          val f = mapping.fields.apply(i)
          f.set(obj, rec.apply(x))
        } }
      }
      catch {
        case e @ (_ : IllegalArgumentException | _ : IllegalAccessException)
            => throw new RuntimeException(e)
      }
    }
  }

}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
