// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.ipfix

import scala.collection.concurrent.TrieMap
import scala.collection.immutable.Queue

/**
  * A `SessionGroup` maintains a set of [[Session Sessions]] for a
  * single transport session.  Put more carefully, it maintains a
  * group of sessions that can be distinguished solely by an
  * observation domain identifier.
  *
  * A SessionGroup is created to create either [[StreamSession
  * StreamSessions]] or [[DatagramSession DatagramSessions]].  A StreamSession
  * is used when reading data from files or over TCP; it supports template
  * withdrawal and re-use of a template ID raises an exception.  A
  * DatagramSession is used when reading data over UDP; it does not support
  * template withdrawal and re-use of template IDs is permitted.
  *
  * `transport` is an object may be used to describe the transport, such as an
  * InetSocketAddress for network transports or a Path for a file transport.
  * It may be null.  The SessionGroup does not use this parameter, and the
  * caller may use it however it chooses.
  *
  *
  * Creates a session group.  Each new session in the group will be
  * created using the given session factory.
  *
  * @param infoModel        the information model
  * @param transport        an object that describes the transport of the
  *                         transport session.  It will be an
  *                         InetSocketAddress for network transports, or
  *                         a Path for a file transport.  May be null.
  * @param streamSessions   `true` when the group is used to read data from a
  *                         file or over TCP (creates a [[StreamSession]]),
  *                         and `false` for reading data over UDP (creates a
  *                         [[DatagramSession]]).
  */
final case class SessionGroup(
  infoModel: InfoModel,
  transport: AnyRef,
  streamSessions: Boolean = true)
{
  /** Template callbacks set on the Group are used to initialize the
    * callbacks when new Sessions are created. */
  @volatile
  private[this] var templateCallbacks = Queue.empty[Session.TemplateCallback]

  /** Sessions in this Group, keyed by the observation domain. */
  private[this] val sessions: TrieMap[Int, Session] = new TrieMap()

  /**
    * Registers a new template callback.
    *
    * A session group may have any number of callbacks registered to
    * it.  Each registered callback is added to new sessions created
    * by this session group.
    *
    * @param callback  the template callback
    */
  def register(callback: Session.TemplateCallback): Unit = {
    templateCallbacks = templateCallbacks :+ callback
  }

  /**
    * Determines whether group is maintaining a session matching the
    * given domain.
    *
    * @param domain  an observation domain identifier
    * @return `true` if this group has a session for the domain
    */
  def hasSession(domain: Int): Boolean = {
    sessions.contains(domain)
  }

  /**
    * Retrieves a session from the group as an Option.
    *
    * @param domain  an observation domain identifier
    * @return the session associated with the domain, or `None` if
    *         there is no such session
    */
  def getSession(domain: Int): Option[Session] = sessions.get(domain)

  /**
    * Retrieves a session from the group.  If a session with the given
    * domain is not found, creates it, adds it, and returns the new
    * session.
    *
    * @param domain  an observation domain identifier
    * @return the session associated with the domain
    */
  def getOrCreateSession(domain: Int): Session = {
    sessions.getOrElseUpdate(domain, {
      val s =
        if ( streamSessions ) {
           new StreamSession(this, domain)
        } else {
          new DatagramSession(this, domain)
        }
      s.register(templateCallbacks)
      s
    })
  }

  /**
    * Removes a session from the group.
    *
    * @param domain  an observation domain identifier
    * @return `true` if the group previously contained a
    *         session for the domain
    */
  def retireSession(domain: Int): Boolean = {
    sessions.remove(domain).nonEmpty
  }

  /**
    * Returns an iterator over the Sessions in the SessionGroup.
    */
  def iterator: Iterator[Session] = sessions.valuesIterator
}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
