// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.ipfix

import scala.collection.mutable.WeakHashMap


/**
  * Extracts the specified field from a [[Record]] and casts the field
  * to a type.  Does not descend into any structured data elements
  * (lists) on the Record.  To descend into sub-records of the Record,
  * use the [[DeepFieldExtractor]].
  *
  * Maintains a mapping from [[Template]] to field position to improve
  * performance when extracting a field from multiple Records that
  * share the same Template.
  *
  * @tparam T The type to cast the field to.
  * @param spec The specifier for the field to extract.
  *
  * @see DeepFieldExtractor
  */
final class SimpleFieldExtractor[T] private (spec: FieldSpec)
    extends FieldExtractor[T]
{
  private[this] val map = WeakHashMap.empty[Template, Int]

  /**
    * Extracts the field from a Record as an Option.
    */
  def extractFrom(record: Record): Option[T] = {
    val template = record.template
    val idx = synchronized {
      map.getOrElseUpdate(template, template.indexOf(spec))
    }
    if ( -1 == idx ) {
      None
    } else {
      Option(record(idx).asInstanceOf[T])
    }
  }
}


/**
  * A [[SimpleFieldExtractor]] factory.
  */
object SimpleFieldExtractor {
  /** Returns a new field extractor that returns the first element that
    * matches an Identifier.
    */
  def apply[T](ident: Identifier): SimpleFieldExtractor[T] =
    new SimpleFieldExtractor(FieldSpec(ident))

  /** Returns a new field extractor that returns the `nth` element that
    * matches an Identifier.
    */
  def apply[T](ident: Identifier, nth: Int): SimpleFieldExtractor[T] =
    new SimpleFieldExtractor(FieldSpec(ident, nth))

  /** Returns a new field extractor that returns the first element whose
    * name matches a String.
    */
  def apply[T](name: String): SimpleFieldExtractor[T] =
    new SimpleFieldExtractor(FieldSpec(name))

  /** Returns a new field extractor that returns the `nth` element whose
    * name matches a String.
    */
  def apply[T](name: String, nth: Int): SimpleFieldExtractor[T] =
    new SimpleFieldExtractor(FieldSpec(name, nth))
}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
