// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.ipfix

import java.io.{EOFException, InputStream}
import java.nio.ByteBuffer
import java.nio.channels.{Channels, FileChannel, ReadableByteChannel}
import java.nio.file.{Path, StandardOpenOption}


/**
  * The StreamMessageReader is used to get the next [[Message]] from a
  * channel.  An instance of this type is normally created by an
  * [[InputStreamSession]] or a [[TCPSession]].
  *
  * @param channel The channel to read.
  * @param sessionGroup The SessionGroup that holds the [[InfoModel]] used to
  * interpret data read from the channel.
  */
final case class StreamMessageReader(
  channel: ReadableByteChannel,
  sessionGroup: SessionGroup)
    extends MessageReader()
{
  private[this] val HEADER_LENGTH = Message.headerLength

  /** Buffer into which data from the channel is read. */
  private[this] val buffer = ByteBuffer.allocate(0x20000)

  // position buffer at limit so remaining is 0
  buffer.position(buffer.limit())

  /** Offset into the input channel for reporting errors */
  private[this] var offset = 0L

  /** Whether the end of the input channel has been reached */
  private[this] var eof = false

  /** Holds an exception encountered during read until all previously read data
    * is processed */
  var pending = Option.empty[Throwable]

  /**
    * Fill `buffer` with as much data from `channel` as possible.  Returns
    * `true` if `buffer` contains at least `minNeeded` octets.
    *
    *  Returns `false` if `buffer` is empty, no more data can be read from
    * `channel`, and there is no pending exception.
    *
    * If `buffer` does not have enough data, throws either the pending
    * exception from reading the channel or an EOFException about an
    * incomplete message.  Prior to throwing the exception, it clears the
    * pending exception and sets the buffer to be empty.
    *
    * @param atHeader whether class is attempting to read a Message header
    * @param minNeeded the number of octets needed for the Message
    */
  private[this] def readBytes(
    atHeader: Boolean,
    minNeeded: Int = HEADER_LENGTH):
      Boolean =
  {
    // attempt to read more data if not at EOF
    if ( !eof && buffer.remaining() < minNeeded ) {
      buffer.compact()
      try {
        do {
          val rv = channel.read(buffer)
          eof = ( -1 == rv )
        } while ( !eof && buffer.position() < minNeeded )
      } catch {
        case e: Exception =>
          pending = Option(e)
          eof = true
      }
      buffer.flip()
    }
    if ( buffer.remaining() >= minNeeded ) {
      true
    } else if ( buffer.remaining() == 0 && pending.isEmpty ) {
      false
    } else {
      // Either throw the pending exception (and clear it) or an EOFException
      // about the remaining bytes in the buffer.  Move the buffer's position
      // to its limit so additional calls to haveNext return false.
      val ex = pending.getOrElse {
        val rem = buffer.remaining()
        val obj =
          if ( atHeader ) { "a Message header" } else { "the Message length" }
        new EOFException(s"Channel's remaining data is shorter than ${obj}" +
          s" (have ${rem} of ${minNeeded} needed) at offset ${offset}")
      }
      pending = None
      buffer.position(buffer.limit())
      throw ex
    }
  }

  /**
    * Returns `true` if `channel` contains enough bytes to contain an IPFIX
    * Message header.  Returns `false` when all data has been read.  Blocks if
    * `channel` does not have the necessary number of bytes and is not at
    * end-of-file.  May throw an error generated from reading the channel or
    * if the channel contains an incomplete Message.
    */
  def hasNext: Boolean = {
    if ( buffer.remaining() >= HEADER_LENGTH ) {
      true
    } else {
      readBytes(true)
    }
  }

  /**
    * Returns the next [[Message]] read from `channel`.  Throws an exception
    * if there is no more data or if there is an issue interpreting the data
    * as a Message.
    *
    * If InvalidIPFIXMessageException is thrown, the buffer's position is not
    * changed, and another call to next() throws the same exception.
    */
  def next(): CollectedMessage = {
    if ( buffer.remaining() < HEADER_LENGTH && !readBytes(true) ) {
      throw new NoSuchElementException("Channel contains no more Messages")
    }

    // check message version
    val version = 0xffff & buffer.getShort(buffer.position())
    if ( version != IPFIX_VERSION ) {
      throw new InvalidIPFIXMessageException(
        f"Incorrect Message version (${version}%#06x," +
          f" expected ${IPFIX_VERSION}%#06x) at offset ${offset}")
    }
    // get and check message length against available data
    val len = 0xffff & buffer.getShort(buffer.position() + 2)
    if ( len < HEADER_LENGTH ) {
      throw new InvalidIPFIXMessageException(
        s"Message specifies an invalid length (got ${len}, expected" +
          s" at least ${HEADER_LENGTH}) at offset ${offset + 2}")
    }
    if ( buffer.remaining() < len ) {
      readBytes(false, len)
    }

    // copy the Message's data into a new buffer
    val buf = ByteBuffer.allocate(len)
    if ( buffer.hasArray() ) {
      buf.put(buffer.array(), buffer.position(), len)
    } else {
      val b = buffer.slice()
      b.limit(len)
      buf.put(b)
    }
    buffer.position(buffer.position() + len)
    offset += len
    CollectedMessage(buf, sessionGroup)
  }

}

/**
  * A [[StreamMessageReader]] factory.
  */
object StreamMessageReader {

  /**
    * A [[StreamMessageReader]] factory.
    *
    * @param channel The channel to read the [[Record Records]] from
    * @param model The information model to use
    */
  def apply(channel: ReadableByteChannel, model: InfoModel):
      StreamMessageReader =
    new StreamMessageReader(channel, SessionGroup(model, channel, true))

  /**
    * A [[StreamMessageReader]] factory.
    *
    * @param path The file path to open for reading the [[Record Records]]
    * @param model The information model to use
    */
  def apply(path: Path, model: InfoModel): StreamMessageReader =
    new StreamMessageReader(
      FileChannel.open(path, StandardOpenOption.READ),
      SessionGroup(model, path, true))

  /**
    * A [[StreamMessageReader]] factory.
    *
    * @param stream The input stream to read the [[Record Records]] from
    * @param model The information model to use
    */
  def apply(stream: InputStream, model: InfoModel): StreamMessageReader =
    new StreamMessageReader(
      Channels.newChannel(stream),
      SessionGroup(model, stream, true))

}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
