// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.ipfix

import java.nio.{ByteBuffer,BufferUnderflowException}
import scala.collection.mutable.ArrayBuffer

//import com.typesafe.scalalogging.StrictLogging

/**
  * An IPFIX template that describes the data in a [[Record]].
  *
  * A template is composed of a sequence of [[InfoElement information
  * elements]] with associated lengths.  (The [[IEFieldSpecifier]] class
  * bundles the [[InfoElement]] and length into a single object.)  An
  * element's length in the template may be different than the "natural"
  * length (reduced length encoding), and an element that supports
  * variable-length values may have a fixed size.
  *
  * If a template has a non-zero scope, the template is an Options
  * Template.
  *
  * When a Template is added to a [[Session]] it is given a unique ID
  * within that Session.  A Template may be added to multiple
  * Sessions; there is not necessarily a relationship among the IDs
  * given to a Template in the different Sessions.
  *
  * @param scope The scope if this is an options template or 0
  * @param elements The elements in the template in the order they occur
  * @param lengths The octet length of the corresponding member of `elements`
  * @param name A name used to identify this Template; [[scala.None$ None]] if
  * not set.
  * @param description A description for this Template; [[scala.None$ None]]
  * not set.
  *
  * @since 1.3.1 The description is set and fixed at Template creation.
  */
final class Template private (
  val scope:            Int,
  private val elements: Array[InfoElement],
  private val lengths:  Array[Int],
  val name:             Option[String],
  val description:      Option[String])
{
  assert(elements.size == lengths.size)

  /**
    * Returns the number of elements in this Template.
    */
  val size: Int = elements.size

  /**
    * Indicates whether the template contains at least one
    * variable-length element.
    */
  lazy val containsVarlen: Boolean = lengths.exists(_ == VARLEN)

  /**
    * Indicates whether the template contains at least one structured
    * data element (that is, a [[BasicList]], [[SubTemplateList]], or
    * [[SubTemplateMultiList]]).
    */
  lazy val containsList: Boolean =
    elements.exists((ie: InfoElement) =>
      ie.dataTypeId match {
        case DataTypes.BasicList | DataTypes.SubTemplateList |
            DataTypes.SubTemplateMultiList
            => true
        case _ => false
      }
    )

  /**
    * Gives the number of octets required to hold the shortest record
    * described by this Template.
    */
  val minimumLength: Int = {
    var sum = 0
    for ( len <- lengths ) {
      sum += (if ( len == VARLEN ) { 1 } else { len })
    }
    sum
  }

  /**
    * Returns the `InfoElement` at position `idx`.
    */
  def apply(idx: Int): InfoElement = elements(idx)

  /**
    * Returns the length of the element at position `idx` as specified
    * by the [[IEFieldSpecifier]].
    */
  def elementLength(idx: Int): Int = lengths(idx)

  /**
    * Returns true if this Template is an information element
    * meta-data template.
    *
    * An information element meta-data template is used to by records
    * to provide the name, ID, data-type, units, semantics, and range
    * of an information element that is not part of the standard
    * information model.  Specifically, the method tests whether the
    * elements `informationElementId` and `informationElementName` are
    * present.
    */
  lazy val isInfoElementMetadataTemplate: Boolean =
    (isOptionsTemplate &&
      hasElementNamed("informationElementName") &&
      hasElementNamed("informationElementId"))

  /**
    * Returns true if this Template is a meta-data template.
    *
    * A meta-data template is used to by records that describe an
    * information element or a template.
    */
  lazy val isMetadataTemplate: Boolean =
    (isInfoElementMetadataTemplate || isTemplateMetadataTemplate)

  /**
    * Returns true if this Template is an options template.
    *
    * An options template is one whose scope is non-zero.
    */
  lazy val isOptionsTemplate: Boolean = (scope != 0)

  /**
    * Returns true if this Template is a template meta-data template.
    *
    * A template meta-data template is used to by records to provide a
    * name and an optional description to a template.  Specifically,
    * the method tests whether the elements `templateId` and
    * `templateName` are present.
    */
  lazy val isTemplateMetadataTemplate: Boolean =
    (isOptionsTemplate &&
      hasElementNamed("templateName") &&
      hasElementNamed("templateId"))

  /**
    * Returns true if this Template is a withdrawal template.
    *
    * A withdrawal template is one that contains no elements.
    */
  lazy val isWithdrawalTemplate: Boolean = (size == 0)

  /**
    * Returns the position of the [[InfoElement]] in this Template that
    * matches the given field specifier or returns -1 if the element
    * is not found.
    */
  def indexOf(spec: FieldSpec): Int = {
    var count = 0
    spec.name match {
      case Some(name) =>
        // name is set
        spec.ident match {
          case Some(ident) =>
            // match name and ident
            elements.indexWhere(ie => {
              (((ie.name == name) || (ie.ident == ident))
                && ({ count += 1; (count == spec.nth) }))
            })
          case None =>
            // match only name
            elements.indexWhere(ie => {
              ((ie.name == name) && ({ count += 1; (count == spec.nth) }))
            })
        }
      case None =>
        // name is not set
        spec.ident match {
          case Some(ident) =>
            // match only ident
            elements.indexWhere(ie => {
              ((ie.ident == ident) && ({ count += 1; (count == spec.nth) }))
            })
          case None =>
            // nothing to match
            -1
        }
    }
  }

  /**
    * Returns the position of the given information element in this
    * template occurring at or after `from` or returns -1 if the
    * element is not found.
    */
  def indexOf(infoElement: InfoElement, from: Int): Int =
    elements.indexWhere(ie => (ie == infoElement), from)

  /**
    * Returns the position of the first occurrence of the given
    * information element in this Template or returns -1 if the
    * element is not found.
    */
  def indexOf(infoElement: InfoElement): Int =
    elements.indexWhere(ie => (ie == infoElement))

  /**
    * Returns true if this Template contains an element that matches the given
    * field specifier.
    * @since 1.3.1
    */
  def contains(spec: FieldSpec): Boolean = indexOf(spec) != -1

  /**
    * Returns true if this Template contains the given InfoElement.
    * @since 1.3.1
    */
  def contains(infoElement: InfoElement): Boolean =
    elements.contains(infoElement)

  /**
    * Returns true if this Template contains *any* of the InfoElements in
    * `ieIterable`.
    * @since 1.3.1
    */
  def containsAny(ieIterable: Iterable[InfoElement]): Boolean =
    ieIterable.exists(ie => (elements.contains(ie)))


  /**
    * Uses this Template to create a new [[Record]] from a message by
    * interpreting the data in the given buffer.
    *
    * On success, the ByteBuffer's psition is at the end of the data
    * for the Record.  If the buffer is too short, the ByteBuffer's
    * position is unchanged and BufferUnderflowException is thrown.
    * If there is an error contructing the record from the data, the
    * ByteBuffer's position is at the end of the data for the record.
    */
  private[ipfix] def readRecord(
    inbuf: ByteBuffer,
    session: Session,
    message: Option[Message]):
      CollectedRecord =
  {
    // To create the record, this function determines the starting
    // position and length of each field (element) in the record.
    if ( inbuf.remaining() < minimumLength ) {
      throw new BufferUnderflowException()
    }
    // the buffer to pass to CollectedRecord
    val buffer = inbuf.slice().asReadOnlyBuffer()
    val slices = new Array[ByteBuffer](size)
    for ( i <- 0 until size ) {
      var len = lengths(i)
      if ( VARLEN == len ) {
        len = 0xff & buffer.get
        if ( 0xff == len ) {
          len = 0xffff & buffer.getShort()
        }
      }
      try {
        slices(i) = buffer.slice()
        slices(i).limit(len)
      } catch {
        case _: IllegalArgumentException =>
          throw new BufferUnderflowException()
      }
      buffer.position(buffer.position() + len)
    }
    // having read the data for record, set the limit of `buffer` to
    // the current position, and move the position of `inbuf` to after
    // the record
    buffer.limit(buffer.position())
    inbuf.position(inbuf.position() + buffer.limit())
    return new CollectedRecord(this, session, buffer, slices, message)
  }


  /**
    * Returns the number of octets required to write this template
    * definition to an IPFIX stream.
    * @see `minimumLength` for the mininum length of a record that
    * uses this template.
    */
  def octetLength: Int = {
    var len = if ( scope > 0 && elements.length > 0 ) { 6 } else { 4 }
    for ( elem <- elements ) {
      len += (if ( 0 == elem.enterpriseId ) { 4 } else { 8 })
    }
    len
  }

  /**
    * Appends the Template to a buffer for writing to an IPFIX stream.
    * Adds the Template to the [[Session]] if necessary.
    *
    * @throws java.nio.BufferOverflowException if there is not enough
    * room in the buffer for the Template.  Because the method makes
    * multiple writes to the buffer, the buffer's position is unknown
    * if an error is thrown.
    */
  def toBuffer(outbuf: ByteBuffer, session: Session): ByteBuffer = {
    val tid = session.getOrAdd(this)
    outbuf.putShort(tid.toShort)
      .putShort(size.toShort)
    if ( scope > 0 && size > 0 ) {
      outbuf.putShort(scope.toShort)
    }
    for ( i <- 0 until size ) {
      if ( 0 == elements(i).enterpriseId ) {
        outbuf.putShort(elements(i).elementId.toShort)
          .putShort(lengths(i).toShort)
      } else {
        outbuf.putShort((elements(i).elementId | 0x8000).toShort)
          .putShort(lengths(i).toShort)
          .putInt(elements(i).enterpriseId.toInt)
      }
    }
    outbuf
  }

  /**
    * Returns an iterator over the elements in the Template.  Each
    * iteration returns a Tuple2 containing an InfoElement and its
    * length in this Template.
    */
  def iterator: Iterator[(InfoElement, Int)] =
    for ( idx <- Iterator.range(0, size) ) yield (elements(idx), lengths(idx))

  override def toString(): String = {
    val sb = new StringBuilder(s"Template(Scope: $scope, Name: '")
    sb.append(name.getOrElse(""))
    (for ( i <- 0 until size ) yield s"${elements(i).name}(${lengths(i)})").
      addString(sb, s"', Elements:(${size})[", ", ", "])")
    sb.mkString
  }

  /**
    * Returns a String describing this Template.  If `expandIEs` is `false`,
    * includes only the name and element length for each IE in the Template
    * (like `toString()`).  If `expandIEs` is `true`, includes a complete
    * description of each IE (including its type, ID, etc).
    */
  def toString(expandIEs: Boolean): String = {
    if ( !expandIEs ) {
      toString()
    } else {
      val sb = new StringBuilder(s"Template(Scope: $scope, Name: '")
      sb.append(name.getOrElse(""))
        (for ( i <- 0 until size ) yield s"[${elements(i)}](${lengths(i)})").
        addString(sb, s"', Elements:(${size})[", ", ", "])")
      sb.mkString
    }
  }

  override def hashCode: Int = {
    // When creating the hashCode, only include the scope,
    // and the set of elements and their lengths.  See equals().
    var result =
      41 * (
        41 + scope
      ) + name.hashCode
    for ( len <- lengths ) result = 41 * result + len
    for ( elem <- elements ) result = 41 * result + elem.hashCode
    result
  }

  //  According to Scala book, no need to define canEqual() since this
  //  is a final class

  /**
    * Determines whether two templates are equal.  To be equal, two
    * templates must have the same scope, the same set of elements,
    * the same element order, and all elements must have the same
    * length.
    */
  override def equals(other: Any): Boolean =
    other match {
      case that: Template =>
        scope == that.scope &&
        elements.size == that.elements.size &&
        name == that.name &&
        elements.sameElements(that.elements) &&
        lengths.sameElements(that.lengths)
      case _ => false
    }


  /**
    * Determines if two templates have the same set of elements in the
    * same order with the same lengths.  This method is similar to
    * `equals()` but it does not consider the scope.
    */
  def hasSameElements(that: Template): Boolean =
    elements.sameElements(that.elements) && lengths.sameElements(that.lengths)

  /**
    * Checks whether this Template has an element with the specified
    * name.
    */
  private[this] def hasElementNamed(name: String): Boolean =
    elements.exists(_.name == name)

}


/**
  * A [[Template]] factory.
  */
object Template
//    extends StrictLogging
{

  /**
    * Checks whether the scope is valid and returns the value if valid or
    * throws an error if it is not.  Helper for the newTemplate() methods
    * below.
    */
  private[this] def checkScope(optScope: Option[Int], numElements: Int): Int =
  {
    optScope match {
      case None => 0
      case Some(scope) =>
        if ( scope <= 0 || scope > numElements ) {
          throw new IllegalTemplateRecordException(
            s"Illegal scope count: $scope")
        }
        scope
    }
  }

  /**
    * Looks up the IEFieldSpecifiers in the InfoModel to create an Array of
    * InfoElements and another containing their lengths.  Helper for the
    * newTemplate() methods below.
    */
  private[this] def newTemplateHelper(
    fields:       TraversableOnce[IEFieldSpecifier],
    scope:        Option[Int],
    model:        InfoModel,
    name:         Option[String],
    description:  Option[String]): Template =
  {
    val elements = ArrayBuffer.empty[InfoElement]
    val lengths = ArrayBuffer.empty[Int]
    for ( spec <- fields ) {
      lengths += spec.length
      elements += model.get(spec.ident).getOrElse(
        try {
          val b = new InfoElementBuilder()
          b.ident = spec.ident
          b.name = "[" + spec.ident.toString + "]"
          b.dataType = DataTypes.getDataType(DataTypes.OctetArray)
          b.build()
        } catch {
          case e @ (_: InvalidInfoElementException
             | _: IllegalInfoElementAttributeException)
              => throw new IllegalTemplateRecordException(e.toString())
        }
      )
    }
    new Template(checkScope(scope, elements.length),
      elements.toArray, lengths.toArray, name, description)
  }

  /**
    * Creates an Array of InfoElements and another of their lengths.  Helper
    * for the newTemplate() methods below.
    */
  private[this] def newTemplateHelper(
    fields:       TraversableOnce[(InfoElement, Int)],
    scope:        Option[Int],
    name:         Option[String],
    description:  Option[String]): Template =
  {
    val elements = ArrayBuffer.empty[InfoElement]
    val lengths = ArrayBuffer.empty[Int]
    for ( f <- fields ) {
      if ( f._2 < 0 || f._2 > VARLEN ) {
        throw new IllegalFieldSpecifierException
      }
      elements += f._1
      lengths += f._2
    }
    new Template(checkScope(scope, elements.length),
      elements.toArray, lengths.toArray, name, description)
  }

  /**
    * Looks up each String in the InfoModel to get an InfoElement, and passes
    * the result to another helper method.  Helper for the newTemplate()
    * methods below.
    */
  // put model first to differentiate from the IEFieldSpecifier version since
  // scala cannot distinguish TraversableOnces by content
  private[this] def newTemplateHelper(
    model:        InfoModel,
    fields:       TraversableOnce[(String, Int)],
    scope:        Option[Int],
    name:         Option[String],
    description:  Option[String]): Template =
  {
    newTemplateHelper(
      fields.map {spec => (model(spec._1), spec._2)}, scope, name, description)
  }

  /**
    * Creates a new `Template` having the [[InfoElement InfoElements]]
    * whose [[Identifier Identifiers]] are specified in `fields`,
    * where `model` is used to map the `Identifier`s to
    * `InfoElement`s.  The fields are inserted into the `Template` in
    * the order they appear in the Iterator.
    *
    * If an IEFieldSpecifier specifies an unknown InfoElement, a placeholder
    * InfoElement of type OctetArray is created to hold the value.
    *
    * @param fields The fields for the new `Template`.
    * @param model The `InfoModel` in which to find the InfoElements
    * specified in `fields`.
    */
  def newTemplate(
    fields:       TraversableOnce[IEFieldSpecifier],
    model:        InfoModel):
      Template =
  {
    newTemplateHelper(fields, None, model, None, None)
  }

  /**
    * Creates a new `Template` having the [[InfoElement InfoElements]]
    * whose [[Identifier Identifiers]] are specified in `fields`,
    * where `model` is used to map the `Identifier`s to
    * `InfoElement`s.  The fields are inserted into the `Template` in
    * the order they appear in the Iterator.
    *
    * If an IEFieldSpecifier specifies an unknown InfoElement, a placeholder
    * InfoElement of type OctetArray is created to hold the value.
    *
    * @param fields The fields for the new `Template`.
    * @param model The `InfoModel` in which to find the InfoElements
    * specified in `fields`.
    * @param name A name to identify the `Template`.
    * @param description An optional description for the `Template`.
    * @since 1.3.1
    */
  def newTemplate(
    fields:       TraversableOnce[IEFieldSpecifier],
    model:        InfoModel,
    name:         String,
    description:  Option[String]):
      Template =
  {
    newTemplateHelper(fields, None, model, Option(name), description)
  }

  /**
    * Creates a new `Template` using the [[InfoElement]] and length
    * pairs specified in `fields`.  The fields are inserted into the
    * `Template` in the order they appear in the Iterator.
    *
    * @param fields The fields for the new `Template`.
    */
  def newTemplate(
    fields:       TraversableOnce[(InfoElement, Int)]):
      Template =
  {
    newTemplateHelper(fields, None, None, None)
  }

  /**
    * Creates a new `Template` using the [[InfoElement]] and length
    * pairs specified in `fields`.  The fields are inserted into the
    * `Template` in the order they appear in the Iterator.
    *
    * @param fields The fields for the new `Template`.
    * @param name A name to identify the `Template`.
    * @param description An optional description for the `Template`.
    * @since 1.3.1
    */
  def newTemplate(
    fields:       TraversableOnce[(InfoElement, Int)],
    name:         String,
    description:  Option[String]):
      Template =
  {
    newTemplateHelper(fields, None, Option(name), description)
  }

  /**
    * Creates a new `Template` by getting from the [[InfoModel]] the
    * [[InfoElement]] whose name is the String member of `fields`.  The fields
    * are inserted into the `Template` in the order they appear in the
    * Iterator.
    *
    * Throws an exception if the InfoElement does not exist.  To avoid this
    * exception, use the version that takes [[IEFieldSpecifier]].
    *
    * @param fields The fields for the new `Template`.
    * @param model The `InfoModel` in which to find the InfoElements
    * specified in `fields`.
    * @since 1.3.1
    */
  // put model first to differentiate from the IEFieldSpecifier version since
  // scala cannot distinguish TraversableOnces by content
  def newTemplate(
    model:        InfoModel,
    fields:       TraversableOnce[(String, Int)]): Template =
  {
    newTemplateHelper(model, fields, None, None, None)
  }

  /**
    * Creates a new `Template` by getting from the [[InfoModel]] the
    * [[InfoElement]] whose name is the String member of `fields`.  The fields
    * are inserted into the `Template` in the order they appear in the
    * Iterator.
    *
    * Throws an exception if the InfoElement does not exist.  To avoid this
    * exception, use the version that takes [[IEFieldSpecifier]].
    *
    * @param fields The fields for the new `Template`.
    * @param model The `InfoModel` in which to find the InfoElements
    * specified in `fields`.
    * @param name A name to identify the `Template`.
    * @param description An optional description for the `Template`.
    * @since 1.3.1
    */
  // put model first to differentiate from the IEFieldSpecifier version since
  // scala cannot distinguish TraversableOnces by content
  def newTemplate(
    model:        InfoModel,
    fields:       TraversableOnce[(String, Int)],
    name:         String,
    description:  Option[String]): Template =
  {
    newTemplateHelper(model, fields, None, Option(name), description)
  }


  /**
    * Creates a new Options `Template`.  The Options `Template` is
    * given the specified scope, and it has the [[InfoElement
    * InfoElements]] whose [[Identifier Identifiers]] are specified in
    * `fields`, where `model` is used to map the `Identifier`s to
    * `InfoElement`s.  The fields are inserted into the Options
    * `Template` in the order they appear in the Iterator.
    *
    * If an IEFieldSpecifier specifies an unknown InfoElement, a placeholder
    * InfoElement of type OctetArray is created to hold the value.
    *
    * @param scope The scope for the new Options `Template`.
    * @param fields The fields for the new Options `Template`.
    * @param model The `InfoModel` in which to find the InfoElements
    * specified in `fields`.
    */
  def newOptionsTemplate(
    scope:        Int,
    fields:       TraversableOnce[IEFieldSpecifier],
    model:        InfoModel): Template =
  {
    newTemplateHelper(fields, Option(scope), model, None, None)
  }

  /**
    * Creates a new Options `Template`.  The Options `Template` is
    * given the specified scope, and it has the [[InfoElement
    * InfoElements]] whose [[Identifier Identifiers]] are specified in
    * `fields`, where `model` is used to map the `Identifier`s to
    * `InfoElement`s.  The fields are inserted into the Options
    * `Template` in the order they appear in the Iterator.
    *
    * If an IEFieldSpecifier specifies an unknown InfoElement, a placeholder
    * InfoElement of type OctetArray is created to hold the value.
    *
    * @param scope The scope for the new Options `Template`.
    * @param fields The fields for the new Options `Template`.
    * @param model The `InfoModel` in which to find the InfoElements
    * specified in `fields`.
    * @param name A name to identify the `Template`.
    * @param description An optional description for the `Template`.
    * @since 1.3.1
    */
  def newOptionsTemplate(
    scope:        Int,
    fields:       TraversableOnce[IEFieldSpecifier],
    model:        InfoModel,
    name:         String,
    description:  Option[String]): Template =
  {
    newTemplateHelper(fields, Option(scope), model, Option(name), description)
  }

  /**
    * Creates a new Options `Template`.  The Options `Template` is
    * given the specified scope, and it uses the [[InfoElement]] and
    * length pairs specified in `fields`.  The fields are inserted
    * into the `Template` in the order they appear in the Iterator.
    *
    * @param scope The scope for the new Options `Template`.
    * @param fields The fields for the new `Template`.
    */
  def newOptionsTemplate(
    scope:        Int,
    fields:       TraversableOnce[(InfoElement, Int)]):
      Template =
  {
    newTemplateHelper(fields, Option(scope), None, None)
  }

  /**
    * Creates a new Options `Template`.  The Options `Template` is
    * given the specified scope, and it uses the [[InfoElement]] and
    * length pairs specified in `fields`.  The fields are inserted
    * into the `Template` in the order they appear in the Iterator.
    *
    * @param scope The scope for the new Options `Template`.
    * @param fields The fields for the new `Template`.
    * @param name A name to identify the `Template`.
    * @param description An optional description for the `Template`.
    * @since 1.3.1
    */
  def newOptionsTemplate(
    scope:        Int,
    fields:       TraversableOnce[(InfoElement, Int)],
    name:         String,
    description:  Option[String]):
      Template =
  {
    newTemplateHelper(fields, Option(scope), Option(name), description)
  }

  /**
    * Creates a new Options `Template` by getting from the [[InfoModel]] the
    * [[InfoElement]] whose name is the String member of `fields`.  The
    * Options `Template` is given the specified scope.  The fields are
    * inserted into the Options `Template` in the order they appear in the
    * Iterator.
    *
    * Throws an exception if the InfoElement does not exist.  To avoid this
    * exception, use the version that takes [[IEFieldSpecifier]].
    *
    * @param scope The scope for the new Options `Template`.
    * @param fields The fields for the new `Template`.
    * @param model The `InfoModel` in which to find the InfoElements
    * specified in `fields`.
    * @since 1.3.1
    */
  // put model before fields to differentiate from the IEFieldSpecifier
  // version since scala cannot distinguish TraversableOnces by content
  def newOptionsTemplate(
    scope:        Int,
    model:        InfoModel,
    fields:       TraversableOnce[(String, Int)]): Template =
  {
    newTemplateHelper(model, fields, Option(scope), None, None)
  }

  /**
    * Creates a new Options `Template` by getting from the [[InfoModel]] the
    * [[InfoElement]] whose name is the String member of `fields`.  The
    * Options `Template` is given the specified scope.  The fields are
    * inserted into the Options `Template` in the order they appear in the
    * Iterator.
    *
    * Throws an exception if the InfoElement does not exist.  To avoid this
    * exception, use the version that takes [[IEFieldSpecifier]].
    *
    * @param scope The scope for the new Options `Template`.
    * @param fields The fields for the new `Template`.
    * @param model The `InfoModel` in which to find the InfoElements
    * specified in `fields`.
    * @param name A name to identify the `Template`.
    * @param description An optional description for the `Template`.
    * @since 1.3.1
    */
  // put model before fields to differentiate from the IEFieldSpecifier
  // version since scala cannot distinguish TraversableOnces by content
  def newOptionsTemplate(
    scope:        Int,
    model:        InfoModel,
    fields:       TraversableOnce[(String, Int)],
    name:         String,
    description:  Option[String]): Template =
  {
    newTemplateHelper(model, fields, Option(scope), Option(name), description)
  }

  /**
    * Creates a new `Template` object by interpreting the data in the
    * specified buffer.  Adds the Template to the [[Session]] of the
    * [[Message]].
    *
    */
  def fromBuffer(b: ByteBuffer, m: Message, isOptionsTemplate: Boolean):
      Template =
  {
    //println("Template::fromBuffer(b, m).  size of buffer is " + b.remaining()
    //  + " and current position is " + b.position())
    if ( b.remaining() < 4 ) {
      throw new IllegalTemplateRecordException(
        s"Not enougth bytes for template (${b.remaining()} of 4)")
    }
    val tid: Int = 0xffff & b.getShort()
    val count: Int = 0xffff & b.getShort()
    //println(s"Template::fromBuffer(b, m).  tid is $tid and count is $count")
    if ( tid < MIN_TEMPLATE_ID ) {
      throw new IllegalTemplateRecordException(
        s"Invalid record template ID: $tid")
    }
    val scope = if ( !isOptionsTemplate || count == 0 ) {
      None
    } else {
      if ( b.remaining() < 2 ) {
        throw new IllegalTemplateRecordException(
          s"Not enougth bytes for template (${b.remaining()} of 2)")
      }
      Option(0xffff & b.getShort())
    }

    // Check the Session for template metadata and use it if available
    val (name, description) =
      m.session.getTemplateMetadata(tid.toInt) match {
        case Some(md) => (Option(md.name),
          if ( md.description.isEmpty ) {None} else {Option(md.description)} )
        case None => (None, None)
      }

    val list = Array.empty[IEFieldSpecifier] ++ (
      for ( _ <- 1 to count ) yield IEFieldSpecifier.fromBuffer(b))

    val template = newTemplateHelper(
      list, scope, m.session.infoModel, name, description)
    m.session.add(template, tid)
    template
  }

}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
