// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.ipfix.testing

/** An IPFIX record, which belongs in a matching IPFIX record set. */
sealed trait Rec extends Encoded

object Rec {

  /** An IPFIX template record, defining a template for the given ID
    * with the given fields.
    */
  case class Template(id: Int, fields: FieldSpecifier*) extends Rec {
    require(id > 255)
    def encodedLength = 4 + fields.iterator.map(_.encodedLength).sum
    def encodedBytes =
      (b2(id) ++ b2(fields.length) ++
        fields.iterator.flatMap(_.encodedBytes))
  }

  /** An IPFIX options template record, defining an options template for
    * the given ID, with the given scope fields and options fields.
    */
  case class OptionsTemplate(
    id: Int, scope: Seq[FieldSpecifier], fields: Seq[FieldSpecifier])
      extends Rec
  {
    require(id > 255)
    def encodedLength =
      (6 + scope.iterator.map(_.encodedLength).sum +
        fields.iterator.map(_.encodedLength).sum)
    def encodedBytes =
      (b2(id) ++ b2(scope.length + fields.length) ++ b2(scope.length) ++
        scope.iterator.flatMap(_.encodedBytes) ++
        fields.iterator.flatMap(_.encodedBytes))
  }

  /** An IPFIX data record, with the given values encoded as contents.
    * Note that this does no work to make sure that the types of the
    * values provided line up with those expected by the template.
    * Make sure that if the template needs a byte, you include a byte
    * and not some other value.
    */
  case class Data(values: Encoded*) extends Rec {
    def encodedLength = values.iterator.map(_.encodedLength).sum
    def encodedBytes = values.iterator.flatMap(_.encodedBytes)
  }

}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
