// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.ipfix

//import util.BufferIterator
import java.nio.ByteBuffer
import java.time.Instant
import scala.collection.mutable.ArrayBuffer

// import com.typesafe.scalalogging.StrictLogging


/** A [[Message]] that only holds what is needed to interpret a [[Record]]. */
private final case class MinimalMessage(
  val octetLength: Int,
  val exportEpochSeconds: Long,
  val sequenceNumber: Long,
  val observationID: Int,
  val session: Session)
    extends Message
{
  def exportEpochMilliseconds: Long = exportEpochSeconds * 1000L
  def exportTime: Instant = Instant.ofEpochSecond(exportEpochSeconds)
  def dataRecordCount: Int = 0
}



/**
  * The CollectedMessage class interprets a [[java.nio.ByteBuffer
  * ByteBuffer]] to provide an [[scala.collection.Iterator Iterator]]
  * over the [[IpfixSet]]s in the buffer.
  *
  * @param transport  The transport used to collect the message
  *
  * @see [[CollectedMessage$ The companion object]] for more details
  */
final class CollectedMessage private (
  minimal: MinimalMessage,
  buffer: ByteBuffer,
  val transport: AnyRef)
    extends Message with Iterable[IpfixSet]
{

  /**
    * The number of records in this message.
    */
  private[this] var totalCount: Int = 0

  /**
    * Process the data in the buffer into Sets and store in an Array.
    *
    * Expand the content of TemplateSets into Templates and store them
    * on the Session.
    *
    * Expand the content of RecordSets into individual Records but do
    * not fully decode the records.  Potential pathological issue: If
    * a Message contains a TemplateSet, a RecordSet that uses one of
    * those Templates, then either a withdrawal template or a
    * different definition for that TemplateID.
    */
  private[this] val sets: Array[IpfixSet] = {
    val ab = ArrayBuffer.empty[IpfixSet]
    buffer.position(Message.headerLength)
    val b = buffer.slice()
    while ( b.remaining() >= IpfixSet.headerLength ) {
      val s: IpfixSet = IpfixSet.fromBuffer(b, minimal)
      ab += s
      s match {
        case _: TemplateSet =>  // nothing to do
        case rs: RecordSet =>   totalCount += rs.size
      }
    }
    ab.toArray
  }

  // Check the sequence numbers in the message and invoke callbacks
  session.noteMessage(this)

  /**
    * Returns the number of data records in this Message.
    *
    * @return the count of data records
    */
  override def dataRecordCount: Int = totalCount

  /**
    * Returns an [[scala.collection.Iterator Iterator]] that may be
    * used to iterate over the [[IpfixSet IpfixSets]] in this Message.
    */
  def iterator: Iterator[IpfixSet] = sets.iterator


  // for the remaining methods, refer to the MinimalMessage

  /**
    * The length of the Message in octets.
    */
  final def octetLength: Int = minimal.octetLength

  /**
    * Gets the export time of the message, as a number of milliseconds
    * since 1970-01-01T00:00:00.
    *
    * @return export time in milliseconds
    */
  def exportEpochMilliseconds: Long = minimal.exportEpochMilliseconds

  /**
    * Gets the export time of the message.
    *
    * @return message export time
    */
  def exportTime: Instant = minimal.exportTime

  /**
    * Gets the export time of the message, as a number of seconds
    * since 1970-01-01T00:00:00.
    *
    * @return export time in seconds
    */
  def exportEpochSeconds: Long = minimal.exportEpochSeconds

  /**
    * Gets the sequence number of the message.  The sequence number is
    * unique per domain, and is the complete number of data records
    * that have been sent in the session prior to this message.  It
    * can be used to calculate statistics about missing data records.
    *
    * @return the sequence number
    */
  def sequenceNumber: Long = minimal.sequenceNumber

  /**
    * Gets the observation domain identifier associated with the
    * message.
    *
    * @return the observation domain id
    */
  def observationID: Int = minimal.observationID

  /**
    * Returns a session that can be used to interpret template ids in
    * this message.
    *
    * @return the session associated with this message
    */
  def session: Session = minimal.session

  override def toString(): String = {
    s"CollectedMessage(Domain: ${0xffffffffL & observationID}, " +
    s"Length: ${octetLength}, Sequence: ${sequenceNumber}, " +
    s"DataRecs: ${dataRecordCount}, Export: ${exportTime})"
  }

}


/**
  * A [[CollectedMessage]] factory.
  */
object CollectedMessage {

  /**
    * A CollectedMessage factory.
    *
    * @param buffer A ByteBuffer to interpret as a Message
    * @param sessionGroup The SessionGroup to use when interpreting the Message
    */
  def apply(buffer: ByteBuffer, sessionGroup: SessionGroup): CollectedMessage =
  {
    val buf: ByteBuffer = buffer.asReadOnlyBuffer()
    assert(buf.limit() >= Message.headerLength)

    buf.rewind()
    val octetLength: Int = 0xffff & buf.getShort(2)
    assert(buf.limit() == octetLength)

    val exportEpochSeconds: Long = 0xffffffffL & buf.getInt(4)
    val sequenceNumber: Long = 0xffffffffL & buf.getInt(8)
    val observationID: Int = buf.getInt(12)

    //logger.trace(
    //  "CollectedMessage: buf.limit()=" + buf.limit() + ";"
    //  + s" octetLength=$octetLength; sequenceNumber=$sequenceNumber;"
    //  + s" exportTime=$exportTime; observationID=$observationID")

    val session = sessionGroup.getOrCreateSession(observationID)

    val mm = MinimalMessage(octetLength, exportEpochSeconds, sequenceNumber,
      observationID, session)

    new CollectedMessage(mm, buf, sessionGroup.transport)
  }

}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
