// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.ipfix

import com.github.ghik.silencer.silent
import java.lang.reflect.{Field => MemberField}
import java.time.Instant

/**
  * An ExportRecord represents the data for an individual [[Record]]
  * that is being built from instances of objects.  The ExportRecord
  * class supports subclasses that use class members to represent the
  * values for the [[InfoElement information elements]] of a
  * [[Template]].
  *
  * When creating a record from a buffer containing serialzed data,
  * the [[CollectedRecord]] class is used.
  *
  * The [[ArrayRecord]] class is similar this class, except
  * [[ArrayRecord]] is concrete and fields are always accessed by
  * numeric position.  The [[ExportRecord]] class allows the caller to
  * reference members by name or by position; the class may have
  * default values for elements.  The [[ExportRecord]] class forces
  * the caller to provide a class member for each element in the
  * Template whereas the elements of [[ArrayRecord]] are represented
  * by an `Array[Any]`.
  *
  * @example The caller extends this class, adds members that
  * represent the elements in the class, annotates those members with
  * the [[IPFIXExtract]] class, and defines a corresponding template.
  * The caller may then instantiate the class and reference members
  * by name.
  * {{{
  * class MyRec(template: Template) extends ExportRecord(template) {
  *   @IPFIXExtract(name = "sourceIPv4Address")
  *   var sip = org.cert.netsa.data.net.IPv4Address("0.0.0.0")
  *   @IPFIXExtract(name = "flowStartTimeMilliseconds")
  *   var stime = java.time.Instant.EPOCH
  *   @IPFIXExtract(name = "packetTotalCount")
  *   var packets: Long = 0L
  *   // may add other members
  * }
  * val template = Template(Seq(
  *   IEFieldSpecifier(model, "sourceIPv4Address", 4),
  *   IEFieldSpecifier(model, "flowStartTimeMilliseconds", 8),
  *   IEFieldSpecifier(model, "packetTotalCount", 8)))
  * val rec = new MyRec(template)
  * rec.sip = org.cert.netsa.data.net.IPv4Address("10.1.2.3")
  * rec.stime = java.time.Instant.now()
  * rec.packets = 365
  * }}}
  *
  * @param template The Template for this Record.
  */
abstract class ExportRecord(template: Template) extends Record(template) {
  /** Returns an array where the index represents the position of an
    * InfoElement in the template and the value is the MemberField of
    * this class that corrsponds to that InfoElement.
    *
    * Specifically, for the first InfoElement in the template, finds
    * an annotatation on a MemberField of this class that matches that
    * InfoElement and stores that MemberField as the first entry in an
    * array; continuing for all InfoElements.
    *
    * @throws java.util.NoSuchElementException if an InfoElement does
    *     not map to a MemberField.
    */
  private[this] def infoElementsToMemberFields(): Array[MemberField] = {
    // the array to return
    val memFields = new Array[MemberField](template.size)

    // a map from a FieldSpec to its position in the Template
    var fieldSpecs = Map.empty[FieldSpec, Int]
    // a map from Identifier to its count in the template
    var identCount = Map.empty[Identifier, Int]
    for (i <- 0 until template.size) {
      val ie = template(i)
      val id = ie.ident
      val nth = identCount.get(id) match {
        case None => 1
        case Some(n) => n + 1
      }
      identCount = identCount + (id -> nth)
      fieldSpecs = fieldSpecs + (FieldSpec(ie.name, nth) -> i)
    }

    // loop over the annotated fields, find corresponding FieldSpec,
    // set value in the 'memFields' array
    for (field <- this.getClass.getDeclaredFields
      if (field.isAnnotationPresent(classOf[IPFIXExtract])))
    {
      val annot: IPFIXExtract = field.getAnnotation(classOf[IPFIXExtract])
      val spec = FieldSpec(annot.name, annot.nth)
      fieldSpecs.get(spec) match {
        case None => // ignore
        case Some(pos) => memFields(pos) = field
      }
    }

    // check for missing fields
    for (i <- 0 until memFields.size) {
      if (Option(memFields(i)).isEmpty) {
        throw new NoSuchElementException(
          s"No member field for template element ${1 + i} ${template(i)}")
      }
    }
    memFields
  }

  /**
    * Array to support apply() and update() where index is a position
    * in the Template and value is the class member that represents it
    */
  private[this] val memberFields = infoElementsToMemberFields()


  // implements Record.apply
  def apply(idx: Int): Any = {
    val field = memberFields(idx)
    if (!field.isAccessible(): @silent) {
      field.setAccessible(true)
    }
    field.get(this)
  }

  /** Updates the value within the record for the field specified by
    * position in the record's template.
    *
    * @throws java.lang.IndexOutOfBoundsException if the index is out of
    *     range.
    */
  def update(idx: Int, obj: Any): Unit = {
    template(idx).dataType.checkType(obj)
    val field = memberFields(idx)
    if (!field.isAccessible(): @silent) {
      field.setAccessible(true)
    }
    field.set(this, obj)
  }

  // implements Record.exportTime
  val exportTime: Option[Instant] = None

  // implements Record.observationDomain
  val observationDomain: Option[Int] = None

  // implements Record.message
  val message: Option[Message] = None

  // implements Record.octetLength
  def octetLength: Int = {
    var len = 0
    for (i <- 0 until template.size) {
      var sz = template.elementLength(i)
      if (sz == VARLEN) {
        val v = apply(i)
        if (Option(v).nonEmpty) {
          sz = template(i).dataType.octetLength(v, sz)
        }
      }
      len += sz
    }
    len
  }

  // implements Record.detach
  def detach(): Record = this

  /**
    * Copies fields from `source` to `this` by position.
    *
    * Calls {{{update(i, source.apply(i))}}} for all `i` less than the
    * `size` of `source` and `this`.
    */
  def copyFieldsByPosition(source: Record): Unit = {
    val count = if ( size < source.size ) { size } else { source.size }
    for (i <- 0 until count) {
      update(i, source.apply(i))
    }
  }

}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
