// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.ipfix

import java.time.Instant


/**
  * An ArrayRecord represents the data for an individual [[Record]]
  * that is being built from instances of objects.  In the ArrayRecord
  * class, fields are always accessed by position, where field `i`
  * represents the data for [[InfoElement]] `i` in the [[Template]].
  *
  * The [[ExportRecord]] class is similar to this class, except
  * [[ExportRecord]] is abstract and allows the subclass to reference
  * information elements by name.
  *
  * When creating a record from a buffer containing serialzed data,
  * the [[CollectedRecord]] class should be used.
  *
  * An ArrayRecord is created from a [[Template]].  The value for
  * each [[InfoElement information element]] in the Record is
  * initially set to its default value (0 for values and `null` for
  * references).  The update() method is used set the values.
  *
  * @example {{{
  * val template = Template.newTemplate(Seq(
  *   IEFieldSpecifier(model, "sourceIPv4Address", 4),
  *   IEFieldSpecifier(model, "flowStartTimeMilliseconds", 8),
  *   IEFieldSpecifier(model, "packetTotalCount", 8)), session)
  * val rec = ArrayRecord(template)
  * rec(0) = org.cert.netsa.data.net.IPv4Address("10.1.2.3")
  * rec(1) = java.time.Instant.now()
  * rec(2) = 365
  * }}}
  *
  * @param template The Template for this Record.
  *
  * @see [[ArrayRecord$ The companion object]] for more details.
  */
class ArrayRecord private (template: Template) extends Record(template) {
  /** The objects that are the values for the fields in this record. The
    * order here matches the order specified in the [[Template]].
    */
  private val values = new Array[Any](template.size)

  // implements Record.apply
  def apply(idx: Int): Any = values.apply(idx)

  /** Updates the value within the record for the field specified by
    * position in the record's template.
    *
    * @throws java.lang.IndexOutOfBoundsException if the index is out of
    *     range.
    */
  def update(idx: Int, obj: Any): Unit = {
    template(idx).dataType.checkType(obj)
    values.update(idx, obj)
  }

  // implements Record.exportTime
  val exportTime: Option[Instant] = None

  // implements Record.observationDomain
  val observationDomain: Option[Int] = None

  // implements Record.message
  val message: Option[Message] = None

  // implements Record.octetLength
  def octetLength: Int = {
    var len = 0
    for (i <- 0 until template.size) {
      val sz = template.elementLength(i)
      if (sz == VARLEN) {
        template(i).dataTypeId match {
          case DataTypes.BasicList | DataTypes.SubTemplateList
             | DataTypes.SubTemplateMultiList =>
            // FIXME
          case DataTypes.OctetArray | DataTypes.String =>
            // FIXME
          case _ =>
            throw new RuntimeException("VARLEN length for unexpected type " +
              template(i).dataTypeId)
        }
      }
      len += sz
    }
    len
  }

  // implements Record.detach
  def detach(): Record = this


  /**
    * Copies fields from `source` to `this` by position.
    *
    * Calls {{{update(i, source.apply(i))}}} for all `i` less than the
    * minimum of either `size` or `source.size`.
    */
  def copyFieldsByPosition(source: Record): Unit = {
    val count = if ( size < source.size ) { size } else { source.size }
    for (i <- 0 until count) {
      update(i, source.apply(i))
    }
  }

}


/**
  * An [[ArrayRecord]] factory.
  */
object ArrayRecord {
  /**
    * Creates an ArrayRecord from a Template and sets all the values
    * to their defaults (0 for values and `null` for references).
    */
  def apply(template: Template): ArrayRecord =
    new ArrayRecord(template)

  /**
    * Creates an ArrayRecord from an existing record.  The Template
    * of the new record is the same as the source record, and the new
    * record's values are set to the values of the existing record.
    */
  def apply(source: Record): ArrayRecord = {
    val rec = new ArrayRecord(source.template)
    rec.copyFieldsByPosition(source)
    rec
  }

}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
