// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.ipfix

import java.io.EOFException
import java.nio.ByteBuffer

/**
  * A ByteBufferMessageReader is used to get the next [[Message]] from
  * a byte buffer.
  *
  * @param buffer The ByteBuffer to interpret as IPFIX messages.
  * @param sessionGroup The SessionGroup that holds the [[InfoModel]] used to
  * interpret data in the buffer.
  */
final case class ByteBufferMessageReader(
  buffer: ByteBuffer,
  sessionGroup: SessionGroup)
    extends MessageReader()
{

  private[this] val HEADER_LENGTH = Message.headerLength

  /**
    * Returns an EOFException specifying that `buffer` contains a partial
    * message.  Sets the remaining bytes in `buffer` to 0 so additional calls
    * to `haveNext` return `false`.
    */
  private[this] def makeEOF(atHeader: Boolean, minNeeded: Int = HEADER_LENGTH):
      Exception =
  {
    val rem = buffer.remaining()
    val pos = buffer.position()
    val obj =
      if ( atHeader ) { "a Message header" }
      else { "the Message length" }

    // move position to limit so calls to `hasNext` will return `false`
    buffer.position(buffer.limit())

    new EOFException(
      s"Buffer's remaining data is shorter than ${obj}" +
        s" (have ${rem} of ${minNeeded} needed) at position ${pos}")
  }

  /**
    * Returns `true` if the remaining bytes in `buffer` are large enough for
    * an IPFIX Message header.  Returns `false` if `buffer` has no remaining
    * data.  Throws an exception if the buffer contains some remaning data but
    * not enough for a header.
    */
  def hasNext: Boolean = {
    if ( buffer.remaining() >= HEADER_LENGTH ) {
      true
    } else if ( buffer.remaining() == 0 ) {
      false
    } else {
      throw makeEOF(true)
    }
  }


  /**
    * Returns the next [[Message]] in `buffer`.  Throws an exception if there
    * is no more data, if there is not enough data, or if there is an issue
    * interpreting the data as a Message.
    *
    * If InvalidIPFIXMessageException is thrown, the buffer's position is not
    * changed, and another call to next() throws the same exception.
    */
  def next(): CollectedMessage = {
    if ( buffer.remaining() < HEADER_LENGTH ) {
      if ( buffer.remaining() == 0 ) {
        throw new NoSuchElementException("Buffer contains no more Messages")
      }
      throw makeEOF(true)
    }

    // check message version
    val version = 0xffff & buffer.getShort(buffer.position())
    if ( version != IPFIX_VERSION ) {
      throw new InvalidIPFIXMessageException(
        f"Incorrect Message version (got ${version}%#06x," +
          f" expected ${IPFIX_VERSION}%#06x) at position ${buffer.position()}")
    }
    // get and check message length against available data
    val len = 0xffff & buffer.getShort(buffer.position() + 2)
    if ( len < HEADER_LENGTH ) {
      throw new InvalidIPFIXMessageException(
        s"Message specifies an invalid length (got ${len}, expected" +
          s" at least ${HEADER_LENGTH}) at position ${buffer.position() + 2}")
    }
    if ( buffer.remaining() < len ) {
      throw makeEOF(false, len)
    }

    // copy the Message's data into a new buffer
    val buf = ByteBuffer.allocate(len)
    if ( buffer.hasArray() ) {
      buf.put(buffer.array(), buffer.position(), len)
    } else {
      val b = buffer.slice()
      b.limit(len)
      buf.put(b)
    }
    buffer.position(buffer.position() + len)
    CollectedMessage(buf, sessionGroup)
  }

}


object ByteBufferMessageReader {

  /**
    * A [[StreamMessageReader]] factory.
    *
    * @param stream The input stream to read the [[Record Records]] from
    * @param model The information model to use
    */
  def apply(buffer: ByteBuffer, model: InfoModel): ByteBufferMessageReader =
    new ByteBufferMessageReader(buffer, SessionGroup(model, buffer, true))

}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
