// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.ipfix

//import scala.collection.mutable.ArrayBuffer

import java.nio.ByteBuffer
import java.time.Instant
//import util.ListView


/**
  * A CollectedRecord represents the data for an individual [[Record]]
  * that has been created when a [[Template]] interprets a ByteBuffer.
  *
  * To create a Record from existing objects, use the [[ArrayRecord]]
  * class or extend the [[ExportRecord]] abstract class.
  *
  * @param template The template used to interpret the data.
  * @param session The session object from which the ByteBuffers in
  * `data` were read.
  * @param buffer The complete buffer for this record.
  * @param data An array containing pointers into `buffer`, one for each field
  * in the Record.
  * @param message The message from which the record was read.
  *
  * @see [[CollectedRecord$ The companion object]] for more details
  */
final class CollectedRecord private[ipfix] (
  template: Template,
  val session: Session,
  private[this] var buffer: ByteBuffer,
  private[this] var data: Array[ByteBuffer],
  override val message: Option[Message] = None)
    extends Record(template)
{
  private[this] val cached = new Array[Any](size)

  // implements Record.apply
  override def apply(idx: Int): Any = {
    Option(cached(idx)) match {
      case Some(v) => v
      case None =>
        val ie: InfoElement = template(idx)
        cached(idx) = ie.dataType.getValue(data(idx), session, ie)
        cached(idx)
    }
  }

  // implements Record.exportTime
  override def exportTime: Option[Instant] =
    message.map { m => m.exportTime }

  // implements Record.observationDomain
  override def observationDomain: Option[Int] =
    message.map { m => m.observationID }

  // implements Record.octetLength
  def octetLength: Int = buffer.limit()

  // `true` when this Record owns its `buffer`, `false` if `buffer` is part of
  // a larger IPFIX Set or Message
  private[this] var isDetached = false

  // implements Record.detach
  override def detach(): Record = {
    if ( !isDetached ) {
      val buf = ByteBuffer.allocate(buffer.limit())
      val slices = new Array[ByteBuffer](size)

      buffer.position(0)
      buf.put(buffer)

      buf.position(0)
      for ( (d,i) <- data.zipWithIndex ) {
        slices(i) = buf.slice()
        slices(i).limit(d.limit())
        buf.position(buf.position() + d.limit())
      }

      buffer = buf
      data = slices
      isDetached = true
    }
    this
  }

  /**
    * Appends this Record to a buffer for writing to an IPFIX stream.
    * The function uses the template IDs in `session` if the Record
    * contains SubTemplateLists or a SubTemplateMultiList.  Assumes
    * the [[Template Template(s)]] used by the Record have already
    * been added to the Session and appended to the buffer.
    */
  override def toBuffer(outbuf: ByteBuffer, session: Session): ByteBuffer = {
    if ( template.containsList ) {
      super.toBuffer(outbuf, session)
    } else {
      // copy bytes from buffer to outbuf without decoding
      buffer.position(0)
      outbuf.put(buffer)
    }
  }

}


/**
  * A [[CollectedRecord]] factory.
  */
object CollectedRecord {

  /**
    * Creates a Record by interpreting the data in a byte buffer
    * according to a template.
    *
    * @param buffer The buffer of data to interpret
    * @param template The template to use to interpret the data
    * @param message The message the contains the buffer
    */
  def fromBuffer(buffer: ByteBuffer, template: Template, message: Message):
      Record =
  {
    template.readRecord(buffer, message.session, Option(message))
  }

}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
