// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.ipfix

import enumeratum.values.{ShortEnum, ShortEnumEntry}

/**
  * IPFIX Information Element Semantics as defined in RFC5610
  *
  * @see [[IESemantics$ The companion object]] for a complete
  * description and the list of values.
  *
  * @param value The short value representing the IE Semantics.
  * @param name The name of the IE Semantics.
  */
sealed abstract class IESemantics(val value: Short, val name: String)
    extends ShortEnumEntry

/**
  * IPFIX Information Element Semantics as defined in RFC5610
  *
  * A description of the semantics of an IPFIX Information Element.
  * These are taken from the data type semantics defined in Section
  * 3.2 of the IPFIX Information Model (RFC5102); see that section for
  * more information on the types described below.  The special value
  * 0x00 (default) is used to note that no semantics apply to the
  * field; it cannot be manipulated by a Collecting Process or File
  * Reader that does not understand it a priori.
  *
  * See https://www.iana.org/assignments/ipfix/ipfix.xhtml#ipfix-information-element-semantics
  *
  * May be converted to and from [[scala.Short Short]] values.
  */
case object IESemantics extends ShortEnum[IESemantics] {

  /**
    * Collection of all known valid information element semantics.
    * @group Members
    */
  lazy val values = findValues
  // Note: lazy to avoid cyclic initialization problem

  /** The Information Element semantic type represented by the given
    * Short value.
    *
    * @throws java.util.NoSuchElementException if the short represents
    *     no known information element semantic.
    */
  def apply(v: Short): IESemantics =
    withValueOpt(v).getOrElse(
      throw new NoSuchElementException(
        s"Unrecognized information element semantic $v")
    )

  /** The IPFIX Information Element semantic type with the name `name`.
    *
    * @throws java.util.NoSuchElementException if the name represents no
    *     known information element semantic.
    */
  def withName(name: String): IESemantics =
    values.find(_.name == name).getOrElse(
      throw new NoSuchElementException(
        s"Unrecognized information element semantic $name")
    )

  /**
    * Finds the IPFIX IE element semantic value whose name is `name`
    * as an Option.
    *
    * The function seaches linearly through the list of semantics.
    *
    * @return The information element semantic represented by `name`.
    */
  def withNameOpt(name: String): Option[IESemantics] =
    values.find(_.name == name)


  /**
    * IPFIX element sematic value denoting no sematics.
    * @group Values
    */
  case object Default extends IESemantics(0, "default")
  /**
    * IPFIX element semantic value denoting a discrete measured value.
    * @group Values
    */
  case object Quantity extends IESemantics(1, "quantity")
  /**
    * IPFIX element semantic value denoting a counter whose initial
    * value is 0.
    * @group Values
    */
  case object TotalCounter extends IESemantics(2, "totalCounter")
  /**
    * IPFIX element semantic value denoting a counter whose initial
    * value is 0 and that is reset to 0 each time it is exported.
    * @group Values
    */
  case object DeltaCounter extends IESemantics(3, "deltaCounter")
  /**
    * IPFIX element semantic value denoting an identifier.
    * @group Values
    */
  case object Identifier extends IESemantics(4, "identifier")
  /**
    * IPFIX element semantic value denoting a set of bit fields.
    * @group Values
    */
  case object Flags extends IESemantics(5, "flags")
  /**
    * IPFIX element semantic value denoting a list.
    * @group Values
    */
  case object List extends IESemantics(6, "list")
  /**
    * IPFIX element semantic value denoting an SNMP counter.
    * @group Values
    */
  case object SnmpCounter extends IESemantics(7, "snmpCounter")
  /**
    * IPFIX element semantic value denoting an SNMP gauge.
    * @group Values
    */
  case object SnmpGauge extends IESemantics(8, "snmpGauge")
}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
