// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.ipfix

import enumeratum.values.{ShortEnum, ShortEnumEntry}

/**
  * IPFIX Information Element Units as defined in RFC5610
  *
  * @see [[IEUnits$ The companion object]] for a complete description
  * and the list of values.
  *
  * @param value The short value representing the Units.
  * @param name The name of the data type.
  */
sealed abstract class IEUnits(val value: Short, val name: String)
    extends ShortEnumEntry

/**
  * IPFIX Information Element Units as defined in RFC5610
  *
  * A description of the units of an IPFIX Information Element.  These
  * correspond to the units implicitly defined in the Information
  * Element definitions in Section 5 of the IPFIX Information Model
  * (RFC5102); see that section for more information on the types
  * described below.  The special value 0x00 (none) is used to note
  * that the field is unitless.
  *
  * See https://www.iana.org/assignments/ipfix/ipfix.xhtml#ipfix-information-element-units
  *
  * May be converted to and from [[scala.Short Short]] values.
  */
case object IEUnits extends ShortEnum[IEUnits] {

  /**
    * Collection of all known valid information element units.
    * @group Members
    */
  lazy val values = findValues
  // Note: lazy to avoid cyclic initialization problem

  /** The Information Element units type associated with this Short
    * value.
    *
    * @throws java.util.NoSuchElementException if the short represents
    *     no known Information Element unit type.
    */
  def apply(v: Short): IEUnits =
    withValueOpt(v).getOrElse(
      throw new NoSuchElementException(
        s"Unrecognized information element unit $v")
    )

  /** The Information Element units type identified by the name `name`.
    *
    * @throws java.util.NoSuchElementException if the name represents no
    *     known information element unit.
    */
  def withName(name: String): IEUnits =
    values.find(_.name == name).getOrElse(
      throw new NoSuchElementException(
        s"Unrecognized information element unit $name")
    )

  /**
    * Finds the IEUnits whose name is `name` as an Option.
    *
    * The function seaches linearly through the list of units.
    *
    * @return The information element unit represented by `name`.
    */
  def withNameOpt(name: String): Option[IEUnits] = values.find(_.name == name)


  /**
    * IPFIX IE unit value denoting no units.
    * @group Values
    */
  case object NONE extends IEUnits(0, "none")
  /**
    * IPFIX IE unit value denoting bits.
    * @group Values
    */
  case object Bits extends IEUnits(1, "bits")
  /**
    * IPFIX IE unit value denoting octets.
    * @group Values
    */
  case object Octets extends IEUnits(2, "octets")
  /**
    * IPFIX IE unit value denoting packets.
    * @group Values
    */
  case object Packets extends IEUnits(3, "packets")
  /**
    * IPFIX IE unit value denoting flow records.
    * @group Values
    */
  case object Flows extends IEUnits(4, "flows")
  /**
    * IPFIX IE unit value denoting seconds.
    * @group Values
    */
  case object Seconds extends IEUnits(5, "seconds")
  /**
    * IPFIX IE unit value denoting milliseconds.
    * @group Values
    */
  case object MilliSeconds extends IEUnits(6, "milliseconds")
  /**
    * IPFIX IE unit value denoting microseconds.
    * @group Values
    */
  case object MicroSeconds extends IEUnits(7, "microseconds")
  /**
    * IPFIX IE unit value denoting nanoseconds.
    * @group Values
    */
  case object NanoSeconds extends IEUnits(8, "nanoseconds")
  /**
    * IPFIX IE unit value denoting 4-octet words for IPv4 header length.
    * @group Values
    */
  case object FourOctetWords extends IEUnits(9, "4-octet words")
  /**
    * IPFIX IE unit value denoting messages for reliability reporting.
    * @group Values
    */
  case object Messages extends IEUnits(10, "messages")
  /**
    * IPFIX IE unit value denoting number of hops for TTL reporting.
    * @group Values
    */
  case object Hops extends IEUnits(11, "hops")
  /**
    * IPFIX IE unit value denoting entries for MPLS label stack.
    * @group Values
    */
  case object Entries extends IEUnits(12, "entries")
  /**
    * IPFIX IE unit value denoting frames.
    * @group Values
    */
  case object Frames extends IEUnits(13, "frames")
  /**
    * IPFIX IE unit value denoting ports.
    * @group Values
    */
  case object Ports extends IEUnits(14, "ports")
  /**
    * IPFIX IE unit value denoting the units of the inferred
    * information element.
    * @group Values
    */
  case object Inferred extends IEUnits(15, "inferred")
}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
