// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.ipfix

import java.lang.Exception

/**
  * Signals that an exception has occurred during IPFIX processing.
  *
  * @param message A description of the problem discovered.
  */
abstract class IPFIXException(message: String)
    extends Exception(message)

/**
  * Signals that the end of file has been reached.
  */
class EndOfFileException()
    extends IPFIXException("End of file")


/**
  * Signals that there are too few bytes for a [[BasicList]] or that
  * the specifier in the list is invalid.
  */
class IllegalBasicListException(message: String = "Illegal basicList")
    extends IllegalListException(message)

class IllegalFieldSpecifierException()
    extends IPFIXException("Illegal field")


/**
  * Signals that an attribute of an [[InfoElement information
  * element]] is invalid.
  */
class IllegalInfoElementAttributeException(message: String)
    extends IPFIXException(message)


/**
  * Signals an error processing a [[ListElement]].  A parent class for
  * more-specific list exceptions.
  */
class IllegalListException(message: String)
    extends IPFIXException(message)


/**
  * Signals that an [[IpfixSet]] ID is invalid.
  */
class IllegalSetException(message: String = "Illegal set")
    extends IPFIXException(message)


/**
  * Signals that there are not enough bytes for the
  * [[SubTemplateList]] or that the [[Template]] ID for the list is
  * not available.
  */
class IllegalSubTemplateListException(
  message: String = "Illegal subTemplateList")
    extends IllegalListException(message)


/**
  * Signals that there are not enough bytes for the
  * [[SubTemplateMultiList]] or that a length specified within the
  * list is too short.
  */
class IllegalSubTemplateMultiListException(
  message: String = "Illegal subTemplateMultiList")
    extends IllegalListException(message)


/**
  * Signals that a [[Template]] record is invalid.
  */
class IllegalTemplateRecordException(
  message: String = "Illegal template record")
    extends IPFIXException(message)



/**
  * Signals that an [[InfoElement information element]] is invalid.
  */
class InvalidInfoElementException(message: String)
    extends IPFIXException(message)


/**
  * Signals that an exception has occurred during IPFIX processing.
  *
  * @param message A description of the problem discovered.
  */
class InvalidIPFIXMessageException(message: String)
    extends IPFIXException(message)


/**
  * Signals an attempt to put more octets into a Message than the
  * IPFIX protocol allows.
  */
class MessageTooLongException(
  message: String = "Message length execeeds 65535 octets")
    extends IPFIXException(message)

/**
  * Signals that a [[Session]] does not contain a [[Template]].
  */
class NoTemplateException(
  message: String = "This session does not contain the template")
    extends IPFIXException(message)
{
  // auxiliary constructor
  /** Signals that a [[Session]] does not contain a [[Template]] having
    * the given ID. */
  def this(tid: Int) =
    this(s"This session does not contain a template having ID ${tid}")

  // auxiliary constructor
  /** Signals that a [[Session]] does not contain the given [[Template]]
    * (when attempting to find its ID). */
  def this(template: Template) =
    this(s"This session does not contain the template ${template}")
}


/**
  * Signals that all possible Template IDs have been used.
  */
class OutOfTemplateIdsException()
    extends IPFIXException("Out of template ids")


/**
  * Signals that a [[ListElement]] may not be added to a [[Record]] or
  * a sub-Record may not be added to a List because they have
  * different [[Session]] objects.
  */
class SessionMismatchException(
  message: String = "Session mismatch adding object to container")
    extends IPFIXException(message)
{
  /* auxiliary constructor */
  def this(objectType: String, containerType: String) =
    this(s"Session mismatch adding $objectType to $containerType")
}

/**
  * Signals that there is not enough data available to read the
  * [[Message]].
  */
class TruncatedReadException(message: String)
    extends IPFIXException(message)
{
  /* auxiliary constructor */
  def this(objType: String, length: Int, remaining: Int) =
    this(s"$objType is truncated (bytes: $length, remaining: $remaining)")
}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
