// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.ipfix

/**
  * Information in messages of the IPFIX protocol is modeled in terms
  * of Information Elements of the IPFIX [[InfoModel information model]].  The
  * InfoElement class represents a single Information Element.
  *
  * Typically an InfoElement is created either by parsing an XML file
  * that describes a set of Information Elements or by using the
  * [[InfoElementBuilder]] class.
  *
  * @param name A unique and meaningful name for the Element.
  * @param ident A numeric identifier of the Element.
  * @param dataTypesValue The type of data stored in this Element.
  * @param group The semantics and applicability of the Element.
  * @param semanticsValue A qualification of the integral value
  * @param description A description of the Element.
  * @param unitsValue The unit for the measure of the Element, if applicable.
  * @param rangeMin The lower bound of the valid range for the Element.
  * @param rangeMax The upper bound of the valid range for the Element.
  *
  * @throws InvalidInfoElementException if name's length is 0, if
  * rangeMin is greater than rangeMax, or if the elementId and
  * enterpriseId of the ident are both 0.
  *
  * @see [[InfoElement$ The companion object]] for more details
  */
final case class InfoElement private (
  val name: String,
  val ident: Identifier,
  private val dataTypesValue: Short,
  val group: String,
  private val semanticsValue: Short,
  val description: String,
  private val unitsValue: Short,
  val rangeMin: Long,
  val rangeMax: Long)
{
  if (0 == name.length) {
    throw new InvalidInfoElementException("name is missing or has length of 0")
  }
  if (0 == ident.elementId && 0 == ident.enterpriseId) {
    throw new InvalidInfoElementException(
      "elementId and enterpriseId are both 0")
  }
  if (rangeMin > rangeMax) {
    throw new InvalidInfoElementException("rangeMin is greater than rangeMax")
  }

  // Initially, the constructor used the DataType, IESemanatics, and
  // IEUnits types directly, but that caused problems when attempting
  // to unserialize (unknown constructor DataType$String$).
  //
  // I changed the constructor to use the Short values instead, and
  // used "@transient val" when defining the following, but then the
  // value was null in the unserialized object.  An attempt to set the
  // "val" in the "readObject()" method failed since they are
  // read-only.  I considered making them "var"s, but they should be
  // constant.  So I have now made them "def"s.  I hope there is not
  // too much overhead in calling these.

  /**
    * An alternate representation of the dataType of the Element.
    */
  def dataTypeId: DataTypes = DataTypes.withValue(dataTypesValue)

  /**
    * The type of data stored in this element.
    */
  def dataType: DataType = DataTypes.getDataType(dataTypeId)

  /**
    * A qualification of the integral value.
    */
  def semantics: IESemantics = IESemantics.withValue(semanticsValue)

  /**
    * The unit for the measure of the Element, if applicable.
    */
  def units: IEUnits = IEUnits.withValue(unitsValue)

  /**
    * The elementId part of the Element's [[Identifier]].
    */
  def elementId: Int = ident.elementId

  /**
    * The enterpriseId part of the Element's [[Identifier]], or 0 if
    * none.
    */
  def enterpriseId: Long = ident.enterpriseId

  override def toString(): String = {
    val id = if (enterpriseId != 0L) {
      s"${enterpriseId}/${elementId}"
    } else {
      s"${elementId}"
    }
    s"InfoElement(${name}, ${id}, ${dataTypeId}, ${semantics}, ${units})"
  }

}


/**
  * An [[InfoElement]] factory.
  */
object InfoElement {

  private[this] val rangeRegex =
    (raw"(0x\p{XDigit}+|\p{Digit}+) *- *(0x\p{XDigit}+|\p{Digit}+)".r).unanchored

  /**
    * Takes a set of string parameters (typically derived by parsing
    * XML) and creates an InfoElement instance.
    *
    * Note: parameters are ascii-betically sorted
    */
  private[this] def makeElement(
    dataType: String,
    dataTypeSemantics: String,
    description: String,
    elementId: String,
    enterpriseId: String,
    group: String,
    name: String,
    range: String,
    units: String): InfoElement =
  {
    //println("InfoElemt::makeElement('" + dataType + "', '" +
    //  dataTypeSemantics + "', <description> + '" + elementId + "', '" +
    //  enterpriseId + "', '" + group + "', '" + name + "', '" + range +
    //  "', '" + units + "')")

    val id =
      if (enterpriseId.isEmpty) {
        Identifier(elementId.toInt)
      } else {
        Identifier(elementId.toInt, enterpriseId.toLong)
      }
    val typeSemantics: Short =
      if (dataTypeSemantics.isEmpty) {
        IESemantics.Default.value
      } else {
        IESemantics.withName(dataTypeSemantics).value
      }
    val dtype: Short =
      if (dataType.isEmpty) {
        DataTypes.OctetArray.value
      } else {
        DataTypes.withName(dataType).value
      }
    val ieUnits: Short =
      if (units.isEmpty) {
        IEUnits.NONE.value
      } else {
        // Not only is IPFIX write-only its documentation is as well,
        // and that is why IANA has units of both "entries" (IE 22)
        // and "label stack entries" (IE 202).  And also "4 octets"
        // (IE 207) instead of "4-octet words", but at least this one
        // is consistent (because there is only one of them).
        try {
          IEUnits.withName(units).value
        } catch {
          case e: NoSuchElementException => {
            units match {
              case _ => throw e
            }
          }
        }
      }
    val (min: Long, max: Long) =
      range match {
        case "" => (0L, 0L)
        case rangeRegex(lo, hi) =>
          (java.lang.Long.decode(lo), java.lang.Long.decode(hi))
        case _ =>
          println("odd range " + range)
          (0L, 0L)
      }

    new InfoElement(name, id, dtype, group, typeSemantics,
      description, ieUnits, min, max)
  }

  /**
    * Parses XML representing a single Information Element and creates
    * a new InfoElement instance.
    */
  def fromXML(node: scala.xml.Node): InfoElement = {
    if (node.label != "record") {
      throw new InvalidInfoElementException(
        "expected XML label 'record', got" + node.label)
    }
    try {
      makeElement(
        (node \ "dataType").text,
        (node \ "dataTypeSemantics").text,
        (node \ "description").text,
        (node \ "elementId").text,
        (node \ "enterpriseId").text,
        (node \ "group").text,
        (node \ "name").text,
        (node \ "range").text,
        (node \ "units").text)
    } catch {
      case e @ (_ : NumberFormatException | _ : NoSuchElementException)
          => throw new IllegalInfoElementAttributeException(e.toString())
    }
  }

  /**
    * Uses the values on an [[InfoElementMetadata]] to create a new
    * InfoElement.
    */
  def fromInfoElementMetadata(metadata: InfoElementMetadata): InfoElement = {
    val ieBuilder = new InfoElementBuilder()
    ieBuilder.dataType = metadata.ieDataType
    ieBuilder.description = metadata.description
    ieBuilder.elementId = metadata.id
    ieBuilder.enterpriseId = metadata.pen
    ieBuilder.name = metadata.name
    ieBuilder.rangeMin = metadata.rangeBegin
    ieBuilder.rangeMax = metadata.rangeEnd
    ieBuilder.semantics = metadata.semantics
    ieBuilder.units = metadata.units
    ieBuilder.build()
  }

  protected[ipfix] def apply(b: InfoElementBuilder): InfoElement = {
    if ( b.name.isEmpty ) {
      throw new InvalidInfoElementException("InfoElement's name is empty")
    }
    if ( b.ident.isEmpty ) {
      throw new InvalidInfoElementException("InfoElement's Identifier is empty")
    }
    if ( b.dataType.isEmpty ) {
      throw new InvalidInfoElementException("InfoElement's dataType is empty")
    }
    new InfoElement(b.name.get, b.ident.get, b.dataType.get.id.value,
      b.group.getOrElse(""), b.semantics.getOrElse(IESemantics.Default).value,
      b.description.getOrElse(""), b.units.getOrElse(IEUnits.NONE).value,
      b.rangeMin.getOrElse(0), b.rangeMax.getOrElse(0))
  }
}



/*  ****************************************************************** */

/**
  * The InfoElementBuilder class is used to create new [[InfoElement
  * InfoElements]].
  *
  * An empty builder may be created, or a builder may be created based
  * on an existing InfoElement.
  */
final class InfoElementBuilder() {
  private var optDataType: Option[DataType] = None
  private var optDescription: Option[String] = None
  private var optGroup: Option[String] = None
  private var optIdent: Option[Identifier] = None
  private var optName: Option[String] = None
  private var optRangeMax: Option[Long] = None
  private var optRangeMin: Option[Long] = None
  private var optSemantics: Option[IESemantics] = None
  private var optUnits: Option[IEUnits] = None

  /**
    * Uses the settings of the builder to create a new InfoElement.
    */
  def build(): InfoElement = InfoElement(this)

  /**
    * Returns the element's dataType as an Option.
    */
  def dataType: Option[DataType] = optDataType

  /**
    * Sets the element's dataType.
    */
  def dataType_=(dataType: DataType): Unit = {
    optDataType = Option(dataType)
  }

  /**
    * Returns the element's description as an Option.
    */
  def description: Option[String] = optDescription

  /**
    * Sets the element's description.
    */
  def description_=(description: String): Unit = {
    optDescription = Option(description)
  }

  /**
    * Returns the elementId portion of the element's ident as an Option.
    */
  def elementId: Option[Int] = optIdent.map { id => id.elementId }

  /**
    * Sets the elementId portion of the element's ident.  The
    * enterpriseId is unchanged or initialized to 0 if none was set.
    */
  def elementId_=(id: Int): Unit = {
    optIdent =
      optIdent match {
        case None => Option(Identifier(id))
        case Some(identifier) => Option(Identifier(id, identifier.enterpriseId))
      }
  }

  /**
    * Returns the enterpriseId portion of the element's ident as an Option.
    */
  def enterpriseId: Option[Long] = optIdent.map { id => id.enterpriseId }

  /**
    * Sets the enterpriseId portion of the element's ident.  The
    * elementId is unchanged or initialized to 0 if none was set.
    */
  def enterpriseId_=(id: Long): Unit = {
    optIdent =
      optIdent match {
        case None => Option(Identifier(0, id))
        case Some(identifier) => Option(Identifier(identifier.elementId, id))
      }
  }

  /**
    * Returns the element's group as an Option.
    */
  def group: Option[String] = optGroup

  /**
    * Sets the element's group.
    */
  def group_=(group: String): Unit = { optGroup = Option(group) }

  /**
    * Returns the element's ident as an Option.
    */
  def ident: Option[Identifier] = optIdent

  /**
    * Sets the element's ident.
    *
    * @throws IllegalInfoElementAttributeException if both the
    *     elementId and enterpriseId are 0.
    */
  def ident_=(ident: Identifier): Unit = {
    if (0 == ident.elementId && 0L == ident.enterpriseId) {
        throw new IllegalInfoElementAttributeException("Illegal ident " + ident)
    }
    optIdent = Option(ident)
  }

  /**
    * Returns the element's name as an Option.
    */
  def name: Option[String] = optName

  /**
    * Sets the element's name.
    */
  def name_=(name: String): Unit = {
    if (0 == name.length) {
      throw new IllegalInfoElementAttributeException("Illegal name \"\"")
    }
    optName = Option(name)
  }

  /**
    * Returns the lower bound of the element's range as an Option.
    */
  def rangeMin: Option[Long] = optRangeMin

  /**
    * Sets the lower bound of the element's range.
    */
  def rangeMin_=(rangeMin: Long): Unit = { optRangeMin = Option(rangeMin) }

  /**
    * Returns the upper bound of the element's range as an Option.
    */
  def rangeMax: Option[Long] = optRangeMax

  /**
    * Sets the upper bound of the element's range.
    */
  def rangeMax_=(rangeMax: Long): Unit = { optRangeMax = Option(rangeMax) }

  /**
    * Returns the element's semantics as an Option.
    */
  def semantics: Option[IESemantics] = optSemantics

  /**
    * Sets the element's semantics.
    *
    * @throws IllegalInfoElementAttributeException if `semantics` does
    *     not map to a valid [[IESemantics]] value.
    */
  def semantics_=(semantics: Int): Unit = {
    try {
      optSemantics = Option(IESemantics(semantics.toShort))
    } catch {
      case _: NoSuchElementException =>
        throw new IllegalInfoElementAttributeException(
          s"Illegal semantics $semantics")
    }
  }

  /**
    * Returns the element's units as an Option.
    */
  def units: Option[IEUnits] = optUnits

  /**
    * Sets the element's units.
    *
    * @throws IllegalInfoElementAttributeException if `units` does not
    *     map to a valid [[IEUnits]] value.
    */
  def units_=(units: Int): Unit = {
    try {
      optUnits = Option(IEUnits(units.toShort))
    } catch {
      case _: NoSuchElementException =>
        throw new IllegalInfoElementAttributeException(s"Illegal units $units")
    }
  }

}


/**
  * An [[InfoElementBuilder]] factory.
  */
object InfoElementBuilder {
  /**
    * Uses the parameters of an existing [[InfoElement Information
    * Element]] to create a new instance of an InfoElementBuilder.
    *
    * @param ie The element to use as the basis for the builder.
    */
  def apply(ie: InfoElement): InfoElementBuilder = {
    val b = new InfoElementBuilder()
    b.optName = Option(ie.name)
    b.optDataType = Option(ie.dataType)
    b.optGroup = Option(ie.group)
    b.optSemantics = Option(ie.semantics)
    b.optIdent = Option(ie.ident)
    b.optDescription = Option(ie.description)
    b.optUnits = Option(ie.units)
    b.optRangeMin = Option(ie.rangeMin)
    b.optRangeMax = Option(ie.rangeMax)
    b
  }
}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
