// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.ipfix

import java.nio.ByteBuffer
import scala.collection.immutable.{Set => ScalaSet}
import scala.collection.mutable.ArrayBuffer


/**
  * A class to represent the contents of an IPFIX SubTemplateList
  * structured data.
  *
  * A SubTemplateList contains a zero or more instances of [[Record
  * Records]] that match a single IPFIX [[Template]].
  *
  * The [[SubTemplateList$ companion object]] has factory methods that
  * return a [[CollectedSubTemplateList]] instance when reading a
  * SubTemplateList or a [[ExportSubTemplateList]] instance when
  * generating a SubTemplateList for writing.
  *
  */
sealed abstract class SubTemplateList protected () extends ListElement()
{
  protected val headerLength = 3

  /**
    * Gets the `Template` that describes the [[Record Records]] in
    * this list.
    */
  def template: Template

  /**
    * Appends the elements in the SubTemplateList to a buffer for
    * writing to an IPFIX stream.  Assumes the SubTemplateList's
    * [[Template]] already has been added to the [[Session]] and
    * written to the buffer.
    */
  def toBuffer(outbuf: ByteBuffer, session: Session): ByteBuffer = {
    // CollectedSubTemplateList overrides this method for conditions
    // when the contents may be copied without decoding
    outbuf.put(semanticId.toByte)
      .putShort(session(template).toShort)
    // encode the elements
    for (rec <- iterator) {
      rec.toBuffer(outbuf, session)
    }
    outbuf
  }

  // implements ListElement.octetLength
  def octetLength: Int = {
    var len = headerLength
    for (rec <- iterator) {
      len += rec.octetLength
    }
    len
  }

  // implements ListElement.allTemplates
  def allTemplates: ScalaSet[Template] = {
    val tmpl = template
    var s = ScalaSet[Template](tmpl)
    if ( tmpl.containsList ) {
      for ( rec <- iterator ) {
        s = s ++ rec.allTemplates
      }
    }
    s
  }

  // implements ListElement.allBasicListElements
  final def allBasicListElements: ScalaSet[InfoElement] = {
    var s = ScalaSet.empty[InfoElement]
    if ( template.containsList ) {
      for ( rec <- iterator ) {
        s = s ++ rec.allBasicListElements
      }
    }
    s
  }

  override def toString(): String = {
    val sb = new StringBuilder()
    iterator.addString(sb, s"STL(Semantics: ${semantics} [", ", ", "])")
    sb.mkString
  }

  override def formatted: String =
    iterator.map(_.formatted).mkString(s"subTemplateList($semantics [", ", ", "])")

}


/**
  * A [[SubTemplateList]] factory.
  */
object SubTemplateList {
  /**
    * Creates a new SubTemplateList by reading data from a buffer.
    *
    * @param buffer The ByteBuffer containing the data representing the list.
    * @param session The IPFIX session from which the buffer was read.
    */
  def fromBuffer(buffer: ByteBuffer, session: Session): SubTemplateList =
    new CollectedSubTemplateList(buffer, session)

  /**
    * Creates a SubTemplateList to which [[Record Records]] may be
    * appended.
    *
    * @param template The Template for each record in this list.
    * @param semantics The semantics for elements of this list.
    */
  def apply(
    template: Template,
    semantics: ListSemantics = ListSemantics.Undefined)
      : SubTemplateList =
  {
    new ExportSubTemplateList(template, semantics)
  }

  /**
    * Creates a SubTemplateList from another SubTemplateList.  When
    * `deep` is `false`, does not do a deep copy of the list; the new
    * list contains references to the same items as the existing list.
    */
  def apply(stl: SubTemplateList, deep: Boolean): SubTemplateList =
    new ExportSubTemplateList(stl, deep)

  /**
    * Creates a SubTemplateList from another SubTemplateList.  Does
    * not do a deep copy of the list; the new list contains references
    * to the same items as the existing list.
    */
  def apply(stl: SubTemplateList): SubTemplateList =
    SubTemplateList.apply(stl, false)

}


/**
  * The CollectedSubTemplateList class is used when reading a
  * SubTemplateList from a data stream.  The class delays realizing
  * the elements of this until they are requested.
  *
  * Use the methods in the [[SubTemplateList]] [[SubTemplateList$
  * companion object]] to create a CollectedSubTemplateList instance.
  *
  * @param buffer The ByteBuffer containing the data representing the list.
  * @param session The IPFIX session from which the buffer was read.
  *
  * @see The [[SubTemplateList$ SubTemplateList companion object]] for
  * a factory method.
  */
final class CollectedSubTemplateList(buffer: ByteBuffer, session: Session)
    extends SubTemplateList()
{
  if (buffer.remaining() < headerLength) {
    throw new IllegalSubTemplateListException(
      "Not enough bytes for subTemplateList " +
        s"(${buffer.remaining()} of ${headerLength})")
  }

  override protected val semanticId = readSemanticId(buffer)

  /**
    * The id of the [[Template]] that describes the [[Record Records]]
    * in this list.
    */
  val tid: Int = 0xffff & buffer.getShort()

  /**
    * Gets the `Template` that describes the [[Record Records]] in
    * this list.  The `Template` is found in the [[Session]]
    * associated with the [[Message]] from which the [[Record]]
    * containing this list was read.
    */
  val template: Template = session.getTemplate(tid).getOrElse {
    throw new IllegalSubTemplateListException(
      s"Cannot get template for id ${tid}")
  }

  /**
    * Appends the elements in the SubTemplateList to a buffer for
    * writing to an IPFIX stream.  Assumes the SubTemplateList's
    * [[Template]] already has been added to the [[Session]] and
    * written to the buffer.
    */
  override def toBuffer(outbuf: ByteBuffer, session: Session): ByteBuffer = {
    if ( template.containsList ) {
      super.toBuffer(outbuf, session)
    } else {
      outbuf.put(semanticId.toByte)
        .putShort(session(template).toShort)
      // copy bytes from buffer to outbuf without decoding
      buffer.position(headerLength)
      outbuf.put(buffer)
    }
  }

  private[this] lazy val recs: Array[Record] = {
    buffer.position(headerLength)
    val ab = ArrayBuffer.empty[CollectedRecord]
    while ( buffer.hasRemaining() ) {
      ab += template.readRecord(buffer, session, None)
    }
    ab.toArray
  }

  // implements ListElement.iterator
  def iterator: Iterator[Record] = recs.iterator

  // implements ListElement.apply
  def apply(idx: Int): Record = recs(idx)

  // implements ListElement.size
  lazy val size: Int = recs.size

}



/**
  * The ExportSubTemplateList class is used to incrementally build a
  * SubTemplateList and export it to a stream.
  *
  * Use the methods in the [[SubTemplateList]] [[SubTemplateList$
  * companion object]] to create an ExportSubTemplateList instance.
  *
  * @param template The Template for each record in this list.
  * @param recs The container for the initial records in this list.
  * @param semanticId The semantics for elements of this list.
  *
  * @see The [[SubTemplateList$ SubTemplateList companion object]] for
  * factory methods.
  */
final class ExportSubTemplateList private (
  val template: Template,
  private[this] val recs: ArrayBuffer[Record],
  protected val semanticId: Short)
    extends SubTemplateList()
{
  /**
    * @param template The Template for each record in this list.
    * @param semantics The semantics for elements of this list.
    */
  def this(template: Template, semantics: ListSemantics) =
    this(template, ArrayBuffer.empty[Record], semantics.value)

  /**
    * Creates a SubTemplateList from another SubTemplateList.  Does
    * not do a deep copy of the list when `deep` is `false`: the new
    * list contains references to the same items as the existing list.
    */
  // getRecords() is in the companion object
  def this(stl: SubTemplateList, deep: Boolean = false) =
    this(stl.template, ExportSubTemplateList.getRecords(stl, deep),
      stl.semantics.value)

  /**
    * Appends a record to the list.  The Record's [[Template]] must be
    * equal to that used by this list.
    */
  def append(rec: Record): ExportSubTemplateList = {
    update(recs.size, rec)
    this
  }

  /**
    * Updates a record in the list.  The valid range for `idx` is 0
    * to the size of this list, inclusive.  The Record's [[Template]]
    * must match that used by this list and the Templates must be in
    * the same [[Session]].
    */
  def update(idx: Int, rec: Record): Unit = {
    if (idx < 0 || idx > recs.size) {
      throw new IndexOutOfBoundsException("bad index")
    }
    if (rec.template != template) {
      throw new SessionMismatchException("Record", "SubTemplateList")
    }
    if (idx == recs.size) {
      recs += rec
    } else {
      recs.update(idx, rec)
    }
  }

  // implements ListElement.iterator
  def iterator: Iterator[Record] = recs.iterator

  // implements ListElement.apply
  def apply(idx: Int): Record = recs(idx)

  // implements ListElement.size
  def size: Int = recs.size

}


/**
  * Contains private methods to support the [[ExportSubTemplateList]] class.
  */
object ExportSubTemplateList {
  /**
    * Returns an ArrayBuffer containing either the Records in the
    * SubTemplateList `stl` when `deep` is `false` or a copy of those
    * Records when `deep` is `true`.  This is a helper method for an
    * auxiliary constructor.
    */
  private def getRecords(stl: SubTemplateList, deep: Boolean):
      ArrayBuffer[Record] =
  {
    val ab = ArrayBuffer.empty[Record]
    ab.sizeHint(stl.size)
    if ( !deep ) {
      ab ++ stl.iterator
    } else {
      ab ++ (for (r <- stl.iterator) yield Record(r, deep))
    }
  }
}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
