// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.ipfix.testing

import java.time.Instant

/** An IPFIX standard message. Contains a sequence of IPFIX record
  * sets of various types. Observation Domain ID is always set to 1
  * for now. Export Time is set to the current time. Sequence Number
  * is allocated in increasing order automatically.
  *
  * Here's an example message:
```
Message(
  Set.Template(
    Rec.Template(0x1001,
      FieldSpecifier(4, 1),    // protocolSpecifier, unsigned8
      FieldSpecifier(85, 8)    // octetTotalCount, unsigned64
    )
  ),
  Set.Data(0x1001, // template described above
    Rec.Data(
      6.toByte,                // TCP protocol number    (byte)
      1234567L                 // 1,234,567 bytes in TCP (long)
    ),
    Rec.Data(
      17.toByte,               // UDP protocol number    (byte)
      9876L                    // 9,876 bytes in UDP     (long)
    )
  )
)
```
  *
  * This does absolutely minimal checking of what's put into the
  * messages, in order to allow testing of malformed messages,
  * messages discouraged by the standard but still allowed, and the
  * like. It's not meant to be the most convenient thing ever, just
  * convenient enough to quickly describe and generate a variety of
  * IPFIX messages for testing.
  */
case class Message(sets: Set[Rec]*) extends Encoded {
  val versionNumber: Int = 0x000a
  val exportTime: Long = Instant.now().getEpochSecond
  val sequenceNumber: Long = Message.nextSequenceNumber()
  val observationDomainId: Long = 1

  def setsBytes = sets.iterator.flatMap(_.encodedBytes)
  def setsLength = sets.iterator.map(_.encodedLength).sum

  def encodedLength = 16 + setsLength
  def encodedBytes =
    (b2(versionNumber) ++ b2(encodedLength) ++
      b4(exportTime) ++ b4(sequenceNumber) ++ b4(observationDomainId) ++
      setsBytes)
}

object Message {
  private var sequenceNumber: Long = 0
  def nextSequenceNumber(): Long = {
    sequenceNumber = sequenceNumber + 1
    sequenceNumber
  }
}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
