// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.ipfix

import java.nio.ByteBuffer

/**
  * The DataType trait defines members that must exist on classes that
  * provide an implementation of each possible [[DataTypes]] value,
  * where those values represent an IPFIX Information Element Data
  * Types as defined in RFC5102.
  */
trait DataType {
  /** A reference to the DataTypes value that this class implements. */
  val id: DataTypes

  /** The name of the DataType. */
  val name: String

  /** The default length of the DataType. */
  val defaultLength: Int

  /** An optional minimum value for the DataType. */
  val minimumValue: Option[Long]

  /** An optional maximum value for the DataType. */
  val maximumValue: Option[Long]

  /**
    * Converts the bytes in the ByteBuffer which represent the
    * specified InfoElement to an object.  `session` is where `buffer`
    * was collected, and it is used to resolve the contents of
    * [[ListElement ListElements]].
    */
  def getValue(buffer: ByteBuffer, session: Session, ie: InfoElement): Any

  /**
    * Returns `true` when `obj` is the correct type for the DataType.
    */
  def checkType(obj: Any): Boolean

  /**
    * Converts the object into a stream of bytes and appends the bytes
    * to the ByteBuffer, where `len` is the length of the
    * [[InfoElement]] according to the record's [[Template]] (that is,
    * the `length` member of the [[IEFieldSpecifier]]).  When the
    * object is a SubTemplateList or SubTemplateMultiList, `session`
    * is used to find the [[Template]] given a Template ID.
    */
  def toBuffer(buffer: ByteBuffer, session: Session, len: Int, obj: Any): ByteBuffer

  /**
    * Returns the number of bytes that toBuffer() would write to the
    * buffer, where `len` is the length of the [[InfoElement]]
    * according to the record's [[Template]] (that is, the `length`
    * member of the [[IEFieldSpecifier]]).  Returns VARLEN when `obj`
    * is not of the correct type for the DataType.  This method is
    * useful primarily for determining the length of variable length
    * data types.
    */
  def octetLength(obj: Any, len: Int): Int
}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
