// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.ipfix

import java.time.Instant


/**
  * An IPFIX stream consists of Messages, which contain Sets (represented by
  * the [[IpfixSet]] class).  The Message trait exposes data that applies to
  * all Sets and [[Record Records]] within that Message.
  *
  * A message is collected as part of a [[Session]].
  */
trait Message {
  /**
    * Gets the export time of the message, as a number of seconds
    * since 1970-01-01T00:00:00.
    *
    * @return export time in seconds
    */
  def exportEpochSeconds: Long

  /**
    * Gets the export time of the message, as a number of milliseconds
    * since 1970-01-01T00:00:00.
    *
    * @return export time in milliseconds
    */
  def exportEpochMilliseconds: Long

  /**
    * Gets the export time of the message.
    *
    * @return message export time
    */
  def exportTime: Instant

  /**
    * Gets the observation domain identifier associated with the
    * message.
    *
    * @return the observation domain id
    */
  def observationID: Int

  /**
    * Gets the sequence number of the message.  The sequence number is
    * unique per domain, and is the complete number of data records
    * that have been sent in the session prior to this message.  It
    * can be used to calculate statistics about missing data records.
    *
    * @return the sequence number
    */
  def sequenceNumber: Long

  /**
    * Returns the number of data records in this Message.
    *
    * @return the count of data records
    */
  def dataRecordCount: Int

  /**
    * Returns a session that can be used to interpret template ids in
    * this message.
    *
    * @return the session associated with this message
    */
  def session: Session
}

/**
  * Holds constants related to the [[Message]] trait.
  */
object Message {
  /**
    * The octet length of an IPFIX Message header.
    */
  val headerLength: Int = 16
}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
