// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io

/**
  * The `ipfix` package provides classes and objects for reading and
  * writing IPFIX data.
  *
  * === Class / Object Overview ===
  *
  * (For a quick overview of the IPFIX format, see the end of this
  * description.)
  *
  * The [[Message]] trait describes the attributes of an IPFIX
  * message, and the [[CollectedMessage]] class and
  * [[CollectedMessage$ object]] are implementations of that trait
  * when reading data.  (Record export does not create specific
  * Message instance.)
  *
  * The [[IpfixSet]] abstract class and [[IpfixSet$ object]] hold the
  * attributes of a Set.  The [[TemplateSet]] class may represent a
  * Template Set or an Options Template Set.
  *
  * The [[Template]] class and [[Template$ object]] are used to
  * represent a Template Record or an Options Template Record.
  *
  * The [[IEFieldSpecifier]] class and [[IEFieldSpecifier$ object]]
  * represent a Field Specifier within an existing Template.  To
  * search for a field within a Template, the user of the `ipfix`
  * package creates a [[FieldSpec]] (the [[FieldSpec$ companion
  * object]]) and attempts to find it within a Template.
  *
  * The Field Specifier uses the numeric [[Identifier]] to identify an
  * Information Element, and an Element is represented by the
  * [[InfoElement]] class and [[InfoElement$ object]].  The
  * [[InfoModel]] class and [[InfoModel$ object]] represent the
  * Information Model.
  *
  * To describe the attributes of an `InfoElement`, several support
  * classes are defined: [[DataTypes$ DataTypes]] is an enumeration
  * that describes the type of data that the element contains, and
  * [[DataType]] is a class that extracts a Field Value with that
  * `DataType`.  [[IESemantics$ IESemantics]] describes the data
  * semantics of an Information Element (e.g., a counter, an
  * identifier, a set of flags), and [[IEUnits$ IEUnits]] describes
  * its units.
  *
  * The Data Set is represented by the [[RecordSet]] class and
  * [[RecordSet$ object]].
  *
  * A Data Record is represented by the [[Record]] abstract class.  This
  * class has three subclasses:
  *   1. The [[CollectedRecord]] class and [[CollectedRecord$
  * object]] are its implementation when reading data.  Its members
  * are always referenced by numeric position.
  *   1. The [[ArrayRecord]] (I do not like this name) and
  * [[ArrayRecord$ object]] may be used to build a Record from Scala
  * objects; its fields are also referenced by numeric position.
  *   1. [[ExportRecord]] is an abstract class that also supports
  * building a Record from Scala objects.  The user extends the class
  * and uses the [[IPFIXExtract]] annotation to mark the members of
  * the subclass that are to be used when writing the Record.
  *
  * A user-defined class that extends [[Fillable]] trait may use the
  * Record's fill() method to copy fields from a Record to the user's
  * class.  It also uses the [[IPFIXExtract]] annotation.
  *
  * A Structured Data Field Value in a Data Record is represented by
  * the [[ListElement]] abstract class.  That abstract class has three
  * abstract subclasses, and each of those has two concrete subclasses
  * (one for reading and one for writing):
  *   1. The [[BasicList]] abstract class ([[BasicList$ object]]) has
  * subclasses [[CollectedBasicList]] and [[ExportBasicList]].
  *   1. The [[SubTemplateList]] abstract class ([[SubTemplateList$
  * object]]) has subclasses [[CollectedSubTemplateList]] and
  * [[ExportSubTemplateList]].
  *   1. The [[SubTemplateMultiList]] abstract class
  * ([[SubTemplateMultiList$ object]]) has subclasses
  * [[CollectedSubTemplateMultiList]] and
  * [[ExportSubTemplateMultiList]].
  *
  * == Reading data ==
  *
  * When reading data, a Record instance is returned by a
  * [[RecordReader]].  The RecordReader uses a class that extends the
  * [[MessageReader]] trait.  The `ipfix` package includes
  * two: [[ByteBufferMessageReader]] and [[StreamMessageReader]].
  *
  * A [[Session]] value represent an IPFIX session, which is part of a
  * [[SessionGroup]].
  *
  * == Writing data ==
  *
  * For writing data, an instance of an [[ExportStream]] must be
  * created using a [[Session]] and the destination
  * [[java.nio.channels.FileChannel FileChannel]].  The user adds
  * Records or Templates to the ExportStream and they are written to
  * the FileChannel.
  *
  *
  * === Overview of IPFIX ===
  *
  * An IPFIX stream is composed of Messages.  Each Message has a
  * 16-byte Message Header followed by one or more Sets.  There are
  * three types of Sets: A Data Set, a Template Set, and an Options
  * Template Set.
  *
  * Each Set has a 4-byte set header followed by one or more Records.
  * A Data Set contains Data Records and a Template Set contains
  * Template Records.
  *
  * A Template Record describes the shape of the data that appears in
  * a Data Record.  A Template Record contains a 4-byte header
  * followed by zero or more Field Specifiers.  Each Field Specifier
  * is either a 4-byte or an 8-byte value that describes a field in
  * the Data Record.
  *
  * A Field Specifier has two parts.  The first is the numeric
  * Information Element Identifier that is defined in an Information
  * Model.  The second is the number of octets the field occupies in
  * the Data Record.
  *
  * A Data Set contains one or more Data Records of the same type,
  * where the type is determined by the Template Record that the Data
  * Set Header refers to.  Each Data Record contains one or more Field
  * Values, where the order and length of the Field Values is given by
  * the Template.
  *
  * A Field Value in a Data Record may be a Structured Data.  There
  * are three types of Structured Data:
  *   1. A Basic List contains one or more instances of a Single
  * Information Element.
  *   1. A SubTemplateList references a single Template ID, and it
  * contains one or more Records that match that Template.
  *   1. The SubTemplateMultiList contains a series of Template IDs
  * and Records that match that Template ID.
  *
  * An IPFIX stream exists in a Transport Session, where a Transport
  * Session is part of a Session Group.  All Sessions in a Session
  * Group use the same Transport Protocol, and only differ in the
  * numeric Observation Domain that is part of the Message Header.
  */
package object ipfix {

  /**
    * The IPFIX version identifier
    */
  final val IPFIX_VERSION: Int = 10

  /**
    * Element size that represents a variable-length element
    */
  final val VARLEN: Int = 0xffff

  /**
    * Set identifier for template sets
    */
  final val TEMPLATE_SET_ID: Int = 2

  /**
    * Set identifier for options template sets
    */
  final val OPTIONS_TEMPLATE_SET_ID: Int = 3

  /**
    * Minimum valid template ID
    */
  final val MIN_TEMPLATE_ID: Int = 0x100

  /**
    * Maximum valid template ID
    */
  final val MAX_TEMPLATE_ID: Int = 0xffff

  /**
    * Maximum number of bytes that can be written to a single IPFIX
    * mesage not including the message header.
    */
  final val EXPORT_BUFFER_USABLE_LEN: Int = 0xffff - Message.headerLength
}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
