// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.ipfix
package util

import java.nio.ByteBuffer


/**
  * An abstract class to assist when Iterating over the contents of a
  * ByteBuffer.
  *
  * This class provides the `hasNext` and `next()` methods.
  * Subclasses must implment the `fromBuffer()` method to process the
  * data and return an object.  The `fromBuffer()` method is only
  * called when the ByteBuffer has at least some minimum number of
  * bytes available.
  *
  * @param buffer The buffer to copy data out of.
  * @param minimum The minimum number of bytes `buffer` must contain
  * in order for the `fromBuffer()` method to be called.
  */
abstract class BufferIterator[T](
  val buffer: ByteBuffer,
  val minimum: Int)
    extends Iterator[T]
{
  /**
    * The next object to return
    */
  private[this] var cached: Option[T] = None

  /**
    * Gets the next object from the buffer.
    */
  protected def fromBuffer(b: ByteBuffer): T

  /**
    * Checks if there are at least `minimum` bytes available in the
    * buffer and if so calls fromBuffer() to get the next item.
    */
  private[this] def getNext(): Boolean = {
    //println(s"BufferIterator::getNext()  minimum is $minimum and"
    //  + " remaining is " + buffer.remaining())
    if (buffer.remaining() < minimum) {
      false
    } else {
      cached = Option(fromBuffer(buffer))
      true
    }
  }

  /**
    * Tests whether this iterator can provide another element.
    *
    * @return `true` if a subsequent call to `next()` will yield an
    * element, `false` otherwise.
    */
  override final def hasNext: Boolean = (cached.nonEmpty || getNext())

  /** The next element for this iterator.
    *
    * @throws java.util.NoSuchElementException when the iterator is
    *     depleted.
    */
  override final def next(): T = {
    if (cached.isEmpty && !getNext()) {
      throw new NoSuchElementException
    }
    val retval: T = cached.get
    cached = None
    retval
  }
}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
