// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.ipfix

import org.cert.netsa.data.net.IPAddress

import java.nio.{ByteBuffer,BufferUnderflowException}
import scala.collection.immutable.{Set => ScalaSet}
import scala.collection.mutable.ArrayBuffer

// import com.typesafe.scalalogging.StrictLogging

/**
  * A class to represent the contents of an IPFIX BasicList structured
  * data.
  *
  * A BasicList contains a zero or more instances of a single IPFIX
  * [[InfoElement Information Element]].
  *
  * Use the [[CollectedBasicList]] subclass when reading a BasicList,
  * and the [[ExportBasicList]] subclass when generating a BasicList
  * for writing.
  *
  * The [[BasicList$ companion object]] has factory methods that
  * return a [[CollectedBasicList]] instance when reading a BasicList
  * or a [[ExportBasicList]] instance when generating a BasicList for
  * writing.
  *
  */
sealed abstract class BasicList protected () extends ListElement()
//    with StrictLogging
{
  /**
    * Gets the identifier for the [[InfoElement information element]]
    * that describes the items in this list.
    */
  def ident: Identifier
  // abstract method

  /**
    * Returns the information element that describes the items in this
    * list.
    */
  val infoElement: InfoElement
  // abstract value

  /**
    * Returns an Iterator over the elements in the list.
    * @tparam T The type to cast the elements to
    */
  def elements[T]: Iterator[T]
  // abstract method

  /**
    * The InfoElement and the length of each element in the list.
    */
  val spec: IEFieldSpecifier
  // abstract value

  /**
    * Appends this BasicList to a buffer for writing to an IPFIX
    * stream.  The function uses the template IDs in `session` if the
    * BasicList contains SubTemplateLists or a SubTemplateMultiList.
    * Assumes those [[Template Templates]] have already been written
    * to the buffer.
    */
  def toBuffer(outbuf: ByteBuffer, session: Session): ByteBuffer = {
    // CollectedBasicList overrides this method for conditions when
    // the contents may be copied without decoding
    if (0 == spec.ident.enterpriseId) {
      outbuf.put(semanticId.toByte)
        .putShort(spec.ident.elementId.toShort)
        .putShort(spec.length.toShort)
    } else {
      outbuf.put(semanticId.toByte)
        .putShort((0x8000 | spec.ident.elementId).toShort)
        .putShort(spec.length.toShort)
        .putInt(spec.ident.enterpriseId.toInt)
    }
    // encode the elements.
    for (e <- elements[Any]) {
      infoElement.dataType.toBuffer(outbuf, session, spec.length, e)
    }
    outbuf
  }


  // implements ListElement.octetLength
  def octetLength: Int = {
    var len = if (0 == spec.ident.enterpriseId) { 5 } else { 9 }
    if (spec.length != VARLEN) {
      len += spec.length * elements.size
    } else {
      for (e <- elements[Any]) {
        len += infoElement.dataType.octetLength(e, spec.length)
      }
    }
    len
  }

  // implements ListElement.allTemplates
  def allTemplates: ScalaSet[Template] = {
    var s = ScalaSet.empty[Template]
    // do not use record iterator; do not want the fake template
    infoElement.dataTypeId match {
      case DataTypes.BasicList =>
        for (e <- elements[BasicList]) {
          s = s ++ e.allTemplates
        }
      case DataTypes.SubTemplateList =>
        for (e <- elements[SubTemplateList]) {
          s = s ++ e.allTemplates
        }
      case DataTypes.SubTemplateMultiList =>
        for (e <- elements[SubTemplateMultiList]) {
          s = s ++ e.allTemplates
        }
      case _ =>
    }
    s
  }

  // implements ListElement.allBasicListElements
  final def allBasicListElements: ScalaSet[InfoElement] = {
    var s = ScalaSet[InfoElement](infoElement)
    // do not use record iterator; do not want the fake template
    infoElement.dataTypeId match {
      case DataTypes.BasicList =>
        for (e <- elements[BasicList]) {
          s = s ++ e.allBasicListElements
        }
      case DataTypes.SubTemplateList =>
        for (e <- elements[SubTemplateList]) {
          s = s ++ e.allBasicListElements
        }
      case DataTypes.SubTemplateMultiList =>
        for (e <- elements[SubTemplateMultiList]) {
          s = s ++ e.allBasicListElements
        }
      case _ =>
    }
    s
  }


  override def toString(): String = {
    val iename: String = infoElement.name
    val sb = new StringBuilder(s"BL(Semantics: ${semantics}, ${iename} = [")
    infoElement.dataTypeId match {
      case DataTypes.String =>
        (for (str <- elements[String]) yield s"(${str.length})$str"
        ).addString(sb, ", ")
      case DataTypes.Boolean =>
        elements[Boolean].addString(sb, ", ")
      case _ @ (DataTypes.Unsigned8 | DataTypes.Unsigned16
         | DataTypes.Signed8 | DataTypes.Signed16) =>
        elements[Int].addString(sb, ", ")
      case _ @ (DataTypes.Unsigned32 | DataTypes.Unsigned64
         | DataTypes.Signed32 | DataTypes.Signed64) =>
        elements[Long].addString(sb, ", ")
      case _ @ (DataTypes.Float32 | DataTypes.Float64) =>
        elements[Double].addString(sb, ", ")
      case _ @ (DataTypes.DateTimeSeconds | DataTypes.DateTimeMicroseconds
         | DataTypes.DateTimeMilliseconds | DataTypes.DateTimeNanoseconds)=>
        elements[java.time.Instant].addString(sb, ", ")
      case _ @ (DataTypes.IPv4Address | DataTypes.IPv6Address) =>
        elements[IPAddress].addString(sb, ", ")
      case _ @ (DataTypes.OctetArray | DataTypes.MacAddress) =>
        elements[util.ListView].addString(sb, ", ")
      case DataTypes.BasicList =>
        elements[CollectedBasicList].addString(sb, ", ")
      case DataTypes.SubTemplateList =>
        elements[SubTemplateList].addString(sb, ", ")
      case DataTypes.SubTemplateMultiList =>
        elements[SubTemplateMultiList].addString(sb, ", ")
    }
    sb.append("])")
    sb.mkString
  }

  override def formatted: String = {
    import org.apache.commons.text.StringEscapeUtils.escapeJson
    def formatValue(v: Any): String = v match {
      case le: ListElement => le.formatted
      case lv: util.ListView => lv.formatted
      case s: String => s"""\"${escapeJson(s)}\""""
      case _ => v.toString()
    }
    elements.map(formatValue)
      .mkString(s"basicList($semantics ${infoElement.name} [", ", ", "])")
  }

}


/**
  * A [[BasicList]] factory and supporting classes.
  */
object BasicList {

  /**
    * Creates a new BasicList by reading data from a buffer.
    *
    * @param buffer The ByteBuffer containing the data representing the list.
    * @param session The IPFIX session from which the buffer was read.
    */
  def fromBuffer(buffer: ByteBuffer, session: Session): BasicList =
    new CollectedBasicList(buffer, session)

  /**
    * Creates a BasicList to which [[Record Records]] may be appended.
    *
    * @param spec The [[Identifier]] of the [[InfoElement]] and the
    * length for each item in this list.
    * @param model The InfoModel in which to find the [[Identifier]]
    * specified in `spec`.
    * @param semantics The semantics for elements of this list.
    * @throws InvalidInfoElementException if `spec` is not a valid InfoElement
    */
  def apply(
    spec: IEFieldSpecifier,
    model: InfoModel,
    semantics: ListSemantics = ListSemantics.Undefined)
      : BasicList =
  {
    val ie = model(spec.ident)
    new ExportBasicList(
      ie, spec.length, ArrayBuffer.empty[Any], semantics.value)
  }

  /**
    * Creates a BasicList from another BasicList.  When `deep` is
    * `false`, does not do a deep copy of the list; the new list
    * contains references to the same items as the existing list.
    * (This is only an issue on the extremely rare chance that the
    * data in the BasicList is a SubTemplateList or
    * SubTemplateMultiList).
    */
  def apply(bl: BasicList, deep: Boolean): BasicList =
    new ExportBasicList(bl, deep)

  /**
    * Creates a BasicList from another BasicList.  Does not do a deep
    * copy of the list; the new list contains references to the same
    * items as the existing list.
    */
  def apply(bl: BasicList): BasicList =
    BasicList.apply(bl, false)
}



/**
  * The CollectedBasicList class is used when reading a BasicList from
  * a data stream.  The class delays realizing the elements of this
  * until they are requested.
  *
  * Use the methods in the [[BasicList]] [[BasicList$ companion
  * object]] to create a CollectedBasicList instance.
  *
  * @param buffer The ByteBuffer containing the data representing the list.
  * @param session The IPFIX session from which the buffer was read.
  *
  * @see The [[BasicList$ BasicList companion object]] for a factory
  * method.
  */
final class CollectedBasicList(buffer: ByteBuffer, val session: Session)
    extends BasicList()
{
  /** Minimum header length needed for a BasicList. */
  private[this] val  minHeaderLen = 5

  if (buffer.remaining() < minHeaderLen) {
    throw new IllegalBasicListException(
      "Not enough bytes for basicList " +
        s"(${buffer.remaining()} of ${minHeaderLen})")
  }

  override protected val semanticId = readSemanticId(buffer)

  // provides BasicList.spec
  val spec: IEFieldSpecifier =
    try {
      IEFieldSpecifier.fromBuffer(buffer)
    } catch {
      case (_ : BufferUnderflowException | _ : IllegalFieldSpecifierException)
          => throw new IllegalBasicListException()
    }

  // buffer is at the end of the header
  private[this] val headerLength = buffer.position()
  assert(5 == headerLength || 9 == headerLength)

  //logger.trace(s"BasicList IESpec=$spec, bufLen=${buffer.remaining()}")

  /**
    * The information element that describes the items in this list.
    */
  lazy val infoElement: InfoElement = {
    session.infoModel.get(spec.ident).getOrElse(
      try {
        val b = new InfoElementBuilder()
        b.ident = spec.ident
        b.name = "[" + spec.ident.toString + "]"
        b.dataType = DataTypes.getDataType(DataTypes.OctetArray)
        b.build()
      } catch {
        case e @ (_: InvalidInfoElementException
           | _: IllegalInfoElementAttributeException)
            => throw new IllegalBasicListException(e.toString())
      }
    )
  }

  /**
    * Gets the identifier for the [[InfoElement information element]]
    * that describes the items in this list.
    */
  lazy val ident: Identifier = spec.ident

  /**
    * Caches the result of parsing the incoming data.
    */
  private[this] lazy val elementAry: Array[Any] = {
    buffer.position(headerLength)
    val ab = ArrayBuffer.empty[Any]
    // create a partially defined function
    val getValue =
      infoElement.dataType.getValue(_: ByteBuffer, session, infoElement)
    var sz = spec.length
    if ( VARLEN != sz ) {
      ab.sizeHint(buffer.remaining() / sz)
      while ( buffer.hasRemaining() ) {
        val b = buffer.slice()
        b.limit(sz)
        ab += getValue(b)
        buffer.position(buffer.position() + sz)
      }
    } else {
      while ( buffer.hasRemaining() ) {
        sz = buffer.get() & 0xff
        if ( 0xff == sz ) {
          sz = buffer.getShort() & 0xffff
        }
        val b = buffer.slice()
        b.limit(sz)
        ab += getValue(b)
        buffer.position(buffer.position() + sz)
      }
    }
    ab.toArray
  }

  // implements BasicList.elements[T]
  def elements[T]: Iterator[T] = elementAry.iterator.asInstanceOf[Iterator[T]]

  // implements ListElement.iterator
  def iterator: Iterator[Record] = recs.iterator

  // implements ListElement.apply
  def apply(idx: Int): Record = recs(idx)

  // implements ListElement.size
  lazy val size: Int = elementAry.size

  private[this] lazy val recs: Array[Record] = {
    // Create a template that has the infoElement has its only member
    val template: Template = Template.newTemplate(
      Iterable((infoElement, spec.length)))
    val ab = ArrayBuffer.empty[Record]
    buffer.position(headerLength)
    while ( buffer.hasRemaining() ) {
      ab += template.readRecord(buffer, session, None)
    }
    ab.toArray
  }

  /**
    * Appends this BasicList to a buffer for writing to an IPFIX
    * stream.  The function uses the template IDs in `session` if the
    * BasicList contains SubTemplateLists or a SubTemplateMultiList.
    * Assumes those [[Template Templates]] have already been written
    * to the buffer.
    */
  override def toBuffer(outbuf: ByteBuffer, session: Session): ByteBuffer = {
    infoElement.dataTypeId match {
      case DataTypes.BasicList | DataTypes.SubTemplateList
         | DataTypes.SubTemplateMultiList =>
        super.toBuffer(outbuf, session)
      case _ =>
        if (0 == spec.ident.enterpriseId) {
          outbuf.put(semanticId.toByte)
            .putShort(spec.ident.elementId.toShort)
            .putShort(spec.length.toShort)
        } else {
          outbuf.put(semanticId.toByte)
            .putShort((0x8000 | spec.ident.elementId).toShort)
            .putShort(spec.length.toShort)
            .putInt(spec.ident.enterpriseId.toInt)
        }
        // copy bytes from buffer to outbuf without decoding
        buffer.position(headerLength)
        outbuf.put(buffer)
    }
  }

}


/**
  * The ExportBasicList class is used to incrementally build a
  * BasicList and export it to a stream.
  *
  * Use the methods in the [[BasicList]] [[BasicList$ companion
  * object]] to create an ExportBasicList instance.
  *
  * @param infoElement The InfoElement that describes the items in this list.
  * @param length The length of each element in the list.
  * @param elementAry The initial elements in the list.
  * @param semanticId The semantics for elements of this list.
  *
  * @see The [[BasicList$ BasicList companion object]] for factory
  * methods.
  */
class ExportBasicList private[ipfix] (
  val infoElement: InfoElement,
  length: Int,
  elementAry: ArrayBuffer[Any],
  protected val semanticId: Short)
    extends BasicList()
{
  override val ident: Identifier = infoElement.ident

  override val spec = IEFieldSpecifier(ident, length)

  // auxiliary contructor
  /**
    * Creates a BasicList from another BasicList.  Does not do a deep
    * copy of the list; the new list contains references to the same
    * items as the existing list.
    */
  def this(bl: BasicList, deep: Boolean) =
    this(bl.infoElement, bl.spec.length, ExportBasicList.copyElements(bl, deep),
      bl.semantics.value)

  /**
    * Appends an element to the list.  The Record's [[Template]] must
    * have a single [[InfoElement]] that matches that used by this
    * list.
    *
    * @return `this`
    */
  def append(rec: Record): ExportBasicList = {
    update(elementAry.size, rec)
    this
  }

  /**
    * Updates an element in the list.  The valid range for `idx` is 0
    * to the size of this list, inclusive.  The Record's [[Template]]
    * must have a single [[InfoElement]] that matches that used by
    * this list.
    */
  def update(idx: Int, rec: Record): Unit = {
    val recTmpl = rec.template
    if (recTmpl.size != 1 || recTmpl(0) != infoElement) {
      throw new SessionMismatchException("Record", "SubTemplateList")
    }
    updateElement(idx, rec.apply(0))
  }

  /**
    * Appends an element to the list.
    *
    * @return `this`
    */
  def appendElement(obj: Any): ExportBasicList = {
    updateElement(elementAry.size, obj)
    this
  }

  /**
    * Updates an element in the list.  The valid range for `idx` is 0
    * to the size of this list, inclusive.
    */
  def updateElement(idx: Int, obj: Any): Unit = {
    if (idx < 0 || idx > elementAry.size) {
      throw new IndexOutOfBoundsException("bad index")
    }
    // invalidate the current set of recs
    recs = None
    infoElement.dataTypeId match {
      case DataTypes.BasicList =>
        val bl = obj match {
          case x: org.cert.netsa.io.ipfix.BasicList => x
          case _ => throw new IllegalFieldSpecifierException
        }
        if (idx == elementAry.size) {
          elementAry += bl
        } else {
          elementAry.update(idx, bl)
        }
      case DataTypes.SubTemplateList =>
        val stl = obj match {
          case x: org.cert.netsa.io.ipfix.SubTemplateList => x
          case _ => throw new IllegalFieldSpecifierException
        }
        if (idx == elementAry.size) {
          elementAry += stl
        } else {
          elementAry.update(idx, stl)
        }
      case DataTypes.SubTemplateMultiList =>
        val stml = obj match {
          case x: org.cert.netsa.io.ipfix.SubTemplateMultiList => x
          case _ => throw new IllegalFieldSpecifierException
        }
        if (idx == elementAry.size) {
          elementAry += stml
        } else {
          elementAry.update(idx, stml)
        }
      case _ =>
        infoElement.dataType.checkType(obj)
        if (idx == elementAry.size) {
          elementAry += obj
        } else {
          elementAry.update(idx, obj)
        }
    }
    ()
  }

  /**
    * Creates a template that has infoElement has its only member.
    */
  private[this] lazy val template: Template = Template.newTemplate(
    Iterable((infoElement, length)))

  // implements BasicList.elements[T]
  def elements[T]: Iterator[T] = elementAry.iterator.asInstanceOf[Iterator[T]]

  // implements ListElement.size
  def size: Int = elementAry.size

  /** Stores the result of converting the elements in this BasicList to
    * Record objects.  When this value is None, it may be initialized
    * by calling the `recsInitialize` method. */
  private[this] var recs = Option.empty[Array[Record]]

  /** Initializes the value of `recs` and returns it. */
  private[this] def recsInitialize: Array[Record] = {
    val v = new Array[Record](elementAry.size)
    for ( i <- 0 until elementAry.size ) {
      val r = ArrayRecord(template)
      r.update(0, elementAry(i))
      v(i) = r
    }
    recs = Option(v)
    v
  }

  // implements ListElement.apply()
  def apply(idx: Int): Record =
    recs.getOrElse(recsInitialize).apply(idx)

  // implements ListElement.iterator
  def iterator: Iterator[Record] =
    recs.getOrElse(recsInitialize).iterator

}


/**
  * Contains private methods to support the [[ExportBasicList]] class.
  */
object ExportBasicList {
  /** Returns the elements in the list `bl`.  This is a helper method
    * for an auxiliary constructor. */
  private def copyElements(bl: BasicList, deep: Boolean): ArrayBuffer[Any] = {
    bl.infoElement.dataTypeId match {
      case DataTypes.BasicList =>
        ArrayBuffer.empty[BasicList] ++ (
          for { e <- bl.elements[BasicList] } yield BasicList(e, deep))
      case DataTypes.SubTemplateList =>
        ArrayBuffer.empty[SubTemplateList] ++ (
          for {
            e <- bl.elements[SubTemplateList]
          } yield SubTemplateList(e, deep))
      case DataTypes.SubTemplateMultiList =>
        ArrayBuffer.empty[SubTemplateMultiList] ++ (
          for {
            e <- bl.elements[SubTemplateMultiList]
          } yield SubTemplateMultiList(e, deep))
      case _ =>
        ArrayBuffer.empty[Any] ++ (for (e <- bl.elements[Any]) yield e)
    }
  }

}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
