// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.ipfix

import scala.collection.mutable.WeakHashMap


/**
  * Extracts a BasicList from a [[Record]].
  *
  * Maintains a mapping from [[Template]] to field position to improve
  * performance when extracting a BasicList from multiple Records that
  * share the same Template.
  *
  * @param ident The [[InfoElement]] Identifier for the field to
  * extract.
  */
final class BasicListFieldExtractor private (ident: Identifier)
    extends FieldExtractor[BasicList]
{
  private val map: WeakHashMap[Template, Int] = new WeakHashMap[Template, Int]()

  /**
    * Extracts the BasicList from a Record as an Option.
    */
  def extractFrom(r: Record): Option[BasicList] = {
    val t = r.template
    map.get(t) match {
      case Some(idx) => Some(r(idx).asInstanceOf[BasicList])
      case None => {
        var bl: Option[BasicList] = None
        var i = 0
        while (i < t.size && bl.isEmpty) {
          val ie = t(i)
          if (ie.dataTypeId == DataTypes.BasicList) {
            val blist = r(i).asInstanceOf[BasicList]
            if (blist.ident == ident) {
              map.put(t, i)
              bl = Some(blist)
            }
          }
          i += 1
        }
        bl
      }
    }
  }
}


/**
  * A [[BasicListFieldExtractor]] factory.
  */
object BasicListFieldExtractor {
  /** Returns a new field extractor that returns the first basic list
    * whose elements match an Identifier.
    */
  def apply(ident: Identifier): BasicListFieldExtractor =
    new BasicListFieldExtractor(ident)
}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
