// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.ipfix.testing

/** An individual IPFIX record set. May be a template set, an options
  * template set, or a data set.
  */
sealed abstract class Set[+T <: Rec](val id: Int) extends Encoded {
  def records: Seq[T]

  def paddingLength: Int = 0

  require(id == 2 || id == 3 || id > 255)

  def recordBytes = records.iterator.flatMap(_.encodedBytes)
  def paddingBytes = Iterator.continually[Byte](0).take(paddingLength)

  def encodedLength =
    4 + records.iterator.map(_.encodedLength).sum + paddingLength
  def encodedBytes =
    b2(id) ++ b2(encodedLength) ++ recordBytes ++ paddingBytes
}

object Set {

  /** An IPFIX template set, containing template records. */
  case class Template(records: Rec.Template*) extends Set[Rec.Template](2)

  /** An IPFIX template set with trailing padding. */
  case class PaddedTemplate(
    override val paddingLength: Int, records: Rec.Template*)
      extends Set[Rec.Template](2)

  /** An IPFIX options template set. */
  case class OptionsTemplate(records: Rec.OptionsTemplate*)
    extends Set[Rec.OptionsTemplate](3)

  /** An IPFIX options template set with trailing padding. */
  case class PaddedOptionsTemplate(
    override val paddingLength: Int, records: Rec.OptionsTemplate*)
      extends Set[Rec.OptionsTemplate](3)

  /** An IPFIX data set for the given template ID. */
  case class Data(override val id: Int, records: Rec.Data*)
    extends Set[Rec.Data](id)
  {
    require(id > 255)
  }

  /** An IPFIX data set with trailing padding. */
  case class PaddedData(
    override val id: Int, override val paddingLength: Int, records: Rec.Data*)
      extends Set[Rec.Data](id)
  {
    require(id > 255)
  }

}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
