// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.silk

import java.io.{OutputStream, DataOutputStream}
import org.apache.hadoop.conf.Configuration

import io.{BufferWriter, LzoOutputStreamBuffer, RawOutputStreamBuffer,
  SnappyOutputStreamBuffer, ZlibOutputStreamBuffer}
import io.BufferUtil._

/**
  * A writer of binary SiLK RWRec files that are readable by SiLK.
  *
  * @example This example reads the contents of "example.rw" and
  * writes it to "copy.rw":
  * {{{
  * val in = new java.io.FileInputStream("example.rw")
  * val out = new java.io.FileOutputStream("copy.rw")
  * val reader = RWRecReader.ofInputStream(in)
  * val writer = RWRecWriter.toOutputStream(out)
  * writer.append(reader)
  * writer.close()
  * }}}
  *
  * @see [[RWRecWriter$ the companion object]] for more details
  */
class RWRecWriter private (
  val out: DataOutputStream,
  val compressionMethod: CompressionMethod,
  val hadoopConfig: Option[Configuration])
{
  /**
    * Set to true once the file's header has been written
    */
  private[this] var headerWritten = false

  /**
    * Length of a single record
    */
  private[this] val recordLength: Short = 88

  /**
    * The size of the buffer to hold data prior to compressing.
    */
  private[this] val bufferSize = (65536 / recordLength) * recordLength

  /**
    * The buffer to hold data prior to compressing.  The append()
    * method fills this buffer.
    */
  private[this] val buffer = Array.ofDim[Byte](bufferSize)

  /**
    * The current offset into the output buffer.
    */
  private[this] var offset = 0

  /**
    * Object use to write/compress the output
    */
  private[this] val writer: BufferWriter =
    compressionMethod match {
      case CompressionMethod.NONE => RawOutputStreamBuffer(out)
      case CompressionMethod.ZLIB => ZlibOutputStreamBuffer(out)
      case CompressionMethod.LZO1X => LzoOutputStreamBuffer(hadoopConfig.get, out)
      case CompressionMethod.SNAPPY => SnappyOutputStreamBuffer(hadoopConfig.get, out)
      case _ => throw new SilkDataFormatException("Unrecognized compression method")
    }


  /**
    * Writes the SiLK file header to the output stream.
    */
  private[this] def writeHeader(): Unit = {
    // create and write the header object
    val header = new Header(Header.BigEndian, FileFormat.FT_RWIPV6ROUTING,
      Header.FileVersion, compressionMethod, SilkVersion(0),
      recordLength, 1, Vector(HeaderEntry.EndOfHeaders))
    header.writeTo(out)
    out.flush()
    headerWritten = true
  }

  /**
    * Stores a single SiLK record `rwrec` into the object's buffer
    * starting at offset.  Does not check that the buffer has enough
    * room for the record.
    */
  private[this] def pack(rwrec: RWRec): Unit =
  {
    putInstant(buffer, offset, rwrec.startTime)
    putElapsed(buffer, offset + 8, rwrec.elapsed)
    putPort(buffer, offset + 12, rwrec.sPort)
    putPort(buffer, offset + 14, rwrec.dPort)
    putProtocol(buffer, offset + 16, rwrec.protocol)
    putFlowType(buffer, offset + 17, rwrec.flowType)
    putSensor(buffer, offset + 18, rwrec.sensor)
    putTCPFlags(buffer, offset + 20, rwrec.flags)
    putTCPFlags(buffer, offset + 21, rwrec.initFlags)
    putTCPFlags(buffer, offset + 22, rwrec.restFlags)
    putTCPState(buffer, offset + 23, rwrec.tcpState)
    putApplication(buffer, offset + 24, rwrec.application)
    putMemo(buffer, offset + 26, rwrec.memo)
    putSNMPInterface16(buffer, offset + 28, rwrec.input)
    putSNMPInterface16(buffer, offset + 30, rwrec.output)
    putPackets32(buffer, offset + 32, rwrec.packets)
    putBytes32(buffer, offset + 36, rwrec.bytes)
    putIPAddress(buffer, offset + 40, rwrec.sIP)
    putIPAddress(buffer, offset + 56, rwrec.dIP)
    putIPAddress(buffer, offset + 72, rwrec.nhIP)
  }

  /**
    * Indicates whether the SiLK file header has been written to the
    * output stream.
    *
    * @return `true` if header has been written.
    */
  def wasHeaderWritten: Boolean = headerWritten


  /**
    * Iterates over the [[org.cert.netsa.io.silk.RWRec RWRecs]] and
    * appends them to the destination stream.  Writes the SiLK file
    * header to the output stream on the first call to this function.
    *
    * This function may be called multiple times.
    */
  def append(iter: Iterator[RWRec]): Unit = {
    if ( !headerWritten ) {
      writeHeader()
    }

    // process the records
    for (rwrec <- iter) {
      if (bufferSize - offset < recordLength) {
        writer.putBuffer(buffer, offset)
        offset = 0
      }
      pack(rwrec)
      offset = offset + recordLength
    }

    if ( offset > 0 ) {
      writer.putBuffer(buffer, offset)
      offset = 0
    }
    out.flush()
  }

  /**
    * Closes the output stream.
    *
    * Writes the SiLK file header to the output stream if it has not
    * been written, writes any buffered records, closes the output
    * stream, and releases resources.
    */
  def close(): Unit = {
    if ( !headerWritten ) {
      writeHeader()
    }
    if ( offset > 0 ) {
      writer.putBuffer(buffer, offset)
      offset = 0
    }
    writer.end()
    out.close()
  }

}


/**
  * The RWRecWriter companion object provides support for creating an
  * [[RWRecWriter]].
  */
object RWRecWriter {

  /**
    * Creates and returns a writer that writes
    * [[org.cert.netsa.io.silk.RWRec RWRecs]] as a binary SiLK RWRec
    * stream to the output stream `s`.  Compresses the output using
    * `compressionMethod`.  Some compression methods require that a
    * hadoop configuration be provided.
    *
    * @throws [[java.util.NoSuchElementException]] when a hadoop
    * configuration is required and none is provided.
    */
  def toOutputStream(
    s: OutputStream,
    compressionMethod: CompressionMethod = CompressionMethod.NONE,
    hadoopConfig: Option[Configuration] = None)
      : RWRecWriter =
  {
    val out = s match {
      case x: DataOutputStream => x
      case y: OutputStream => new DataOutputStream(y)
    }

    new RWRecWriter(out, compressionMethod, hadoopConfig)
  }
}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
