// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.silk
package io

import java.io.{DataInputStream, InputStream}
import org.apache.hadoop.conf.Configuration
import org.apache.hadoop.io.compress.{CompressionCodecFactory, Decompressor}

/**
  * A [[BufferReader]] which reads lzo-compressed data from SiLK file
  * streams, using Hadoop compression codecs.
  */
private[silk] class LzoInputStreamBuffer(
  configuration: Configuration, inStream: InputStream)
    extends BufferReader
{
  private val in: DataInputStream = inStream match {
    case in: DataInputStream => in
    case _ => new DataInputStream(inStream)
  }

  private val decompressor: Decompressor = try {
    val cf = new CompressionCodecFactory(configuration)
    val codec = cf.getCodecByName("lzo")
    codec.createDecompressor()
  } catch {
    case ex: Exception =>
      throw new RuntimeException(
        "Unable to create lzo decompressor--is lzo properly configured?", ex)
  }

  def close(): Unit = {
    in.close()
  }
  def getNextBuffer(): (Array[Byte], Int) = {
    val compressedSize = in.readInt()
    val uncompressedSize = in.readInt()
    val compressedData = Array.ofDim[Byte](compressedSize)
    val uncompressedData = Array.ofDim[Byte](uncompressedSize)
    in.readFully(compressedData)
    decompressor.setInput(compressedData, 0, compressedSize)
    val uncompressedSizeResult =
      decompressor.decompress(uncompressedData, 0, uncompressedSize)
    assert(decompressor.needsInput)
    assert(uncompressedSizeResult == uncompressedSize)
    (uncompressedData, uncompressedSize)
  }
}

private[silk] object LzoInputStreamBuffer {
  def apply(conf: Configuration, inStream: InputStream): LzoInputStreamBuffer =
    new LzoInputStreamBuffer(conf, inStream)
}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
