// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.silk
package io
package ipset

import scala.collection.immutable.Queue

import BufferUtil.{
  getInt8, getIPv6Address}

import org.cert.netsa.data.net.{
  IPv6Address, IPBlock, IPv6Block}


/*
 *    IPSET_REC_VERSION_CIDRBMAP
 *    ==========================
 *
 *    The IPset file format introduced in SiLK-3.7.  The file may
 *    contain either IPv4 or IPv6 addresses (a header entry specifies
 *    which).
 *
 *    IP addresses are stored as "blocks".  Each block starts with an
 *    IP address and a single octet value.  If the single octet is
 *    less than or equal to 0x80, that is the end of the block.  The
 *    IP address is the base address of a netblock and the single
 *    octet is the CIDR mask for the netblock.
 *
 *    If the octet is strictly greater than 0x80, the address is the
 *    base address for a bitmap.  The value of 0x81 means the bitmap
 *    has 256 bits containing a /24 in IPv4 or a /120 in IPv6.  These
 *    bitmaps are similar to those in IPSET_REC_VERSION_CLASSC.
 *
 *    The IPs in the file appear in sorted order.  IPv4 addresses and
 *    values in the bitmap are written in native byte order.  IPv6
 *    addresses are stored as an array of 16 uint8_t's in network byte
 *    order.
 *
 *    The file has a header entry that may be used to determine
 *    whether the file contains IPv4 or IPv6 addresses.  The header
 *    entry is identical with that used by IPSET_REC_VERSION_RADIX.
 *    All fields are 0 except for the leaf length field.  The leaf
 *    length is 4 for files containing IPv4 addresses and 16 for files
 *    containing IPv6.
 */


private[silk] class IPSetV4IPv6Reader(
  reader: BufferReader,
  header: Header
) extends IPSetReader(reader, header)
{
  private[this] val ipLength = 16
  private[this] val recordLength = 1 + ipLength

  private[this] var blocks = Queue[IPBlock]()

  /**
    * If there are more IPBlocks in 'blocks', do nothing.  Otherwise,
    * read the bytes from 'buffer' that represent an IP address and a
    * prefix.  If the prefix is reasonable for the IP address, add a
    * single CIDR block to 'blocks' and return.  If the prefix is the
    * special value cidrBitmapFollows, read the 256-bit bitmap and add
    * one or more IPBlocks to 'blocks'.  If at end of file, do
    * nothing.
    */
  private[this] def getMore() = {
    if ( blocks.isEmpty && checkAvailable(recordLength) ) {
      val ip = getIPv6Address(buffer, bufOffset)
      val prefix = (0xffL & getInt8(buffer, bufOffset + ipLength, swap)).toInt
      bufOffset = bufOffset + recordLength

      if ( prefix != cidrBitmapFollows ) {
        blocks = blocks.enqueue(IPv6Block(ip, prefix))

      } else {
        val shorts = ip.toShorts
        val L =
          for (
            pair <- handleBitmap256(buffer, bufOffset, swap)
          ) yield {
            shorts(7) = ((shorts(7) & 0xff00) | pair(0)).toShort
            IPv6Block(IPv6Address(shorts), lenToCidrV6(pair(1)))
          }
        blocks = blocks ++ L
        bufOffset = bufOffset + bitmap256Length
      }
    }
  }

  override def containsIPv6: Boolean = true

  override def hasNext: Boolean = {
    getMore()
    !blocks.isEmpty
  }

  override def next(): IPBlock = {
    getMore()
    val ip = blocks.head
    blocks = blocks.tail
    ip
  }
}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
