// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.silk
package io
package unpackers

import java.time.{Duration, Instant}
import org.cert.netsa.data.net.{IPv4Address, TCPFlags}

import BufferUtil._

/*
**  FLOWCAP VERSION 6
**  FLOWCAP VERSION 5
**
**  in the following: EXPANDED == ((tcp_state & SK_TCPSTATE_EXPANDED) ? 1 : 0)
**
**    uint32_t      sIP;             //  0- 3  Source IP
**    uint32_t      dIP;             //  4- 7  Destination IP
**
**    uint32_t      bytes;           //  8-11  Byte count
**
**    uint32_t      sTime;           // 12-15  Start time as UNIX epoch secs
**
**    uint16_t      elapsed;         // 16-17  Duration of flow in seconds
**    uint16_t      sPort;           // 18-19  Source port
**
**    uint16_t      dPort;           // 20-21  Destination port
**    uint16_t      service_port;    // 22-23  Port reported by flow collector
**
**    uint16_t      input;           // 24-25  SNMP Input
**    uint16_t      output;          // 26-27  SNMP Output
**
**    uint8_t       pkts[3]          // 28-30  Count of packets
**    uint8_t       proto            // 31     Protocol
**
**    uint8_t       flags            // 32     EXPANDED==0: All TCP Flags
**                                   //        EXPANDED==1: Flags !1st pkt
**    uint8_t       first_flags;     // 33     EXPANDED==0: 0
**                                   //        EXPANDED==1: TCP Flags 1st pkt
**    uint8_t       tcp_state;       // 34     TCP state machine info
**    uint8_t       time_frac[3];    // 35-37  sTime msec & elapsed msec
**
**
**  38 bytes on disk.
*/

private[silk] object FT_FLOWCAP_v5 extends Unpacker {
  val recordLength = 38
  def unpack(
    buffer: Array[Byte], offset: Int, swap: Boolean, header: Header): RWRec =
  {
    val sIP = getIPv4Address(buffer, offset, swap)
    val dIP = getIPv4Address(buffer, offset + 4, swap)
    val bytes = getBytes32(buffer, offset + 8, swap)

    val timeFrac = getInt24(buffer, offset + 35, false)
    val startTimeSecs = getInt32(buffer, offset + 12, swap)
    val elapsedSecs = getInt16(buffer, offset + 16, swap)

    val startTime = Instant.ofEpochMilli(
      Integer.toUnsignedLong(startTimeSecs) * 1000L +
        ((timeFrac >> 14) & 0x000003FF))
    val elapsed = Duration.ofMillis(
      java.lang.Short.toUnsignedInt(elapsedSecs) * 1000 +
        ((timeFrac >> 4) & 0x0000003FF))

    val sPort = getPort(buffer, offset + 18, swap)
    val dPort = getPort(buffer, offset + 20, swap)
    val application = getPort(buffer, offset + 22, swap)
    val input = getSNMPInterface16(buffer, offset + 24, swap)
    val output = getSNMPInterface16(buffer, offset + 26, swap)
    val packets = getPackets24(buffer, offset + 28, swap)
    val protocol = getProtocol(buffer, offset + 31, swap)
    val tcpState = getTCPState(buffer, offset + 34, swap)
    val (flags, initFlags, restFlags) = if ( tcpState.expandedFlags ) {
      val restFlags = getTCPFlags(buffer, offset + 32, swap)
      val initFlags = getTCPFlags(buffer, offset + 33, swap)
      val flags = TCPFlags((restFlags.toByte | initFlags.toByte).toByte)
      (flags, initFlags, restFlags)
    } else {
      (getTCPFlags(buffer, offset + 32, swap), TCPFlags(0), TCPFlags(0))
    }
    // unsupplied
    val flowType = FlowType(0)
    val sensor = Sensor(0)
    val memo: Short = 0
    val nhIP = IPv4Address(0)
    RWRec(
      startTime, elapsed, sPort, dPort, protocol, flowType, sensor,
      flags, initFlags, restFlags, tcpState, application, memo, input,
      output, packets, bytes, sIP, dIP, nhIP).maybeClearTCPStateExpanded
  }
}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
