// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.silk
package io
package unpackers

import java.time.Duration
import org.cert.netsa.data.net.{IPv4Address, SNMPInterface}

import BufferUtil._

/*
**  RWAUGWEB VERSION 5
**
**  in the following: EXPANDED == ((tcp_state & SK_TCPSTATE_EXPANDED) ? 1 : 0)
**
**    uint32_t      rflag_stime;     //  0- 3
**    // uint32_t     rest_flags: 8; //        EXPANDED==0:Empty
**                                   //        EXPANDED==1:TCPflags/!1st pkt
**    // uint32_t     is_tcp    : 1; //        always 1 since all flows TCP
**    // uint32_t     src_is_srv: 1; //        1 if sIP is http server
**    // uint32_t     stime     :22; //        Start time:msec offset from hour
**
**    uint8_t       proto_iflags;    //  4     EXPANDED==0:TCPflags/ALL pkts
**                                   //        EXPANDED==1:TCPflags/1st pkt
**    uint8_t       tcp_state;       //  5     TCP state machine info
**    uint16_t      application;     //  6- 7  Indication of type of traffic
**
**    uint32_t      srvport_elapsed; //  8-11
**    // uint32_t      srv_port : 2; //        Server port: 0=80; 1=443; 2=8080
**    // uint32_t      elapsed  :30; //        Duration of the flow
**
**    uint32_t      pkts;            // 12-15  Count of packets
**    uint32_t      bytes;           // 16-19  Count of bytes
**
**    uint32_t      sIP;             // 20-23  Source IP
**    uint32_t      dIP;             // 24-27  Destination IP
**
**    uint16_t      clnt_port;       // 28-29  Client(non-server) port
**
**
**  30 bytes on disk.
*/

private[silk] object FT_RWAUGWEB_v5 extends Unpacker {
  val recordLength = 30
  def unpack(
    buffer: Array[Byte], offset: Int, swap: Boolean, header: Header): RWRec =
  {
    val (startTime, protocol, tcpState, flags, initFlags, restFlags) =
      unpackTimesFlagsProto(buffer, offset, swap, header)
    val application = getPort(buffer, offset + 6, swap)

    val rflagStime = getInt32(buffer, offset, swap)
    val srvPortElapsed = getInt32(buffer, offset + 8, swap)
    val clientPort = getPort(buffer, offset + 28, swap)
    val serverPort = decodeWWWPort((srvPortElapsed >> 30) & 0x00000003)
    val sourceIsServer = ((rflagStime >> 22) & 0x00000001) != 0
    val (sPort, dPort) = if ( sourceIsServer ) {
      (serverPort, clientPort)
    } else {
      (clientPort, serverPort)
    }
    val elapsed = Duration.ofMillis(srvPortElapsed & 0x3FFFFFFF)

    val packets = getPackets32(buffer, offset + 12, swap)
    val bytes = getBytes32(buffer, offset + 16, swap)
    val sIP = getIPv4Address(buffer, offset + 20, swap)
    val dIP = getIPv4Address(buffer, offset + 24, swap)

    val sensor = header.packedSensorId
    val flowType = header.packedFlowTypeId
    val memo: Short = 0
    val input = SNMPInterface(0)
    val output = SNMPInterface(0)
    val nhIP = IPv4Address(0)
    RWRec(
      startTime, elapsed, sPort, dPort, protocol, flowType, sensor,
      flags, initFlags, restFlags, tcpState, application, memo, input,
      output, packets, bytes, sIP, dIP, nhIP)
  }
}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
