// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.silk
package io
package unpackers

import BufferUtil._

import java.time.Duration
import org.cert.netsa.data.net.{Port, TCPFlags}

/*
**  RWFILTER VERSION 2
**
**    Only the final 12 bits of the final four bytes differ from
**    VERSION 1: the sID is moved all the way to the right and is now
**    8 bits wide, with 4 bits of padding.
**
**    uint32_t      sIP;             //  0- 3  Source IP
**    uint32_t      dIP;             //  4- 7  Destination IP
**
**    uint16_t      sPort;           //  8- 9  Source port
**    uint16_t      dPort;           // 10-11  Destination port
**
**    uint8_t       proto;           // 12     IP protocol
**    uint8_t       flags;           // 13     OR of all TCP flags on all pkts
**    uint8_t       input;           // 14     Router incoming SNMP interface
**    uint8_t       output;          // 15     Router outgoing SNMP interface
**
**    uint32_t      nhIP;            // 16-19  Router Next Hop IP
**    uint32_t      sTime;           // 20-23  Start time of flow-epoch secs
**
**    uint32_t      pef;             // 24-27
**    // uint32_t     pkts      :20; //        Count of packets
**    // uint32_t     elapsed   :11; //        Duration of flow
**    // uint32_t     pflag     : 1; //        'pkts' requires multiplier?
**
**    uint32_t      bbs;             // 28-31
**    // uint32_t     bPPkt     :14; //        Whole bytes-per-packet
**    // uint32_t     bPPFrac   : 6; //        Fractional bytes-per-packet
**    // uint32_t     pad       : 4; //        Padding
**    // uint32_t     sensorID  : 8; //        Sensor ID
**
**
**  32 bytes on disk.
*/

private[silk] object FT_RWFILTER_v2 extends Unpacker {
  val recordLength = 32
  def unpack(
    buffer: Array[Byte], offset: Int, swap: Boolean, header: Header): RWRec =
  {
    val sIP = getIPv4Address(buffer, offset, swap)
    val dIP = getIPv4Address(buffer, offset + 4, swap)
    val sPort = getPort(buffer, offset + 8, swap)
    val dPort = getPort(buffer, offset + 10, swap)
    val protocol = getProtocol(buffer, offset + 12, swap)
    val flags = getTCPFlags(buffer, offset + 13, swap)
    val input = getSNMPInterface8(buffer, offset + 14, swap)
    val output = getSNMPInterface8(buffer, offset + 15, swap)
    val nhIP = getIPv4Address(buffer, offset + 16, swap)
    val startTime = getStartTimeSecs(buffer, offset + 20, swap)
    val pef = getInt32(buffer, offset + 24, swap)
    val inPackets = (pef >> 12) & 0x000FFFFF
    val elapsed = Duration.ofSeconds((pef >> 1) & 0x000007FF)
    val pFlag = (pef & 0x00000001) != 0
    val bbs = getInt32(buffer, offset + 28, swap)
    val bpp = (bbs >> 12) & 0x000FFFFF
    val sensor = Sensor((bbs & 0x000000FF).toShort)
    val (bytes, packets) = unpackBytesPackets(bpp, inPackets, pFlag)
    val flowType = FlowType(0)
    val initFlags = TCPFlags(0)
    val restFlags = TCPFlags(0)
    val tcpState = TCPState(0)
    val application = Port(0)
    val memo: Short = 0
    RWRec(
      startTime, elapsed, sPort, dPort, protocol, flowType, sensor,
      flags, initFlags, restFlags, tcpState, application, memo, input,
      output, packets, bytes, sIP, dIP, nhIP)
  }
}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
