// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.silk

import org.cert.netsa.data.net.{Protocol, Port}

/**
  * A type representing a range of
  * ([[org.cert.netsa.data.net.Protocol]], [[org.cert.netsa.data.net.Port]])
  * Pairs.
  *
  * This is the type of the key when using an
  * [[scala.collection.Iterator Iterator]] over the contents of a
  * PrefixMap file that contains (Protocol, Port) Pairs.
  *
  * @see [[PrefixMapReader]]
  */
final case class PrefixMapProtocolPortPair(
  val startProtocol: Protocol,
  val startPort:     Port,
  val endProtocol:   Protocol,
  val endPort:       Port)
{
  /** return true when i < j, false otherwise */
  private[this] def unsignedCompare(i: Long, j: Long) = (i < j) ^ (i < 0) ^ (j < 0)

  require(!unsignedCompare(endProtocol.toByte, startProtocol.toByte),
    s"startProtocol is greater than endProtocol (${startProtocol.toByte}"
      + s" > ${endProtocol.toByte})")

  require(startProtocol.toByte != endProtocol.toByte ||
          !unsignedCompare(endPort.toShort, startPort.toShort),
    s"startPort is greater than endPort (${startPort.toShort}" +
      s" > ${endPort.toShort})")
}

/**
  * The PrefixMapProtocolPortPair object provides support for creating
  * a [[PrefixMapProtocolPortPair]].
  */
object PrefixMapProtocolPortPair {
  /**
    * Creates a PrefixMapProtocolPortPair from four integers without
    * requiring the caller to create the appropriate type from each.
    */
  def apply(loProto: Int, loPort: Int, hiProto: Int, hiPort: Int): PrefixMapProtocolPortPair =
    new PrefixMapProtocolPortPair(Protocol(loProto), Port(loPort), Protocol(hiProto), Port(hiPort))
}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
