// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.silk

/**
  * A SiLK TCP state flag vector, encoding various properties of TCP
  * packets in a TCP flow, as encoded in a [[scala.Byte Byte]] value.
  *
  * @param toByte The byte value representing this state vector.
  */
final case class TCPState(val toByte: Byte) extends AnyVal {
  /**
    * True if this record contains expanded flag information
    * (`initFlags` and `restFlags`).
    * @group Flags
    */
  def expandedFlags: Boolean = (toByte & TCPState.BIT_EXPANDED_FLAGS) != 0
  /**
    * True if additional packets were seen following a packet with the
    * FIN flag set.
    * @group Flags
    */
  def finFollowed: Boolean = (toByte & TCPState.BIT_FIN_FOLLOWED) != 0
  /**
    * True if all packets in the flow had the same size in bytes.
    * @group Flags
    */
  def uniformPacketSize: Boolean =
    (toByte & TCPState.BIT_UNIFORM_PACKET_SIZE) != 0
  /**
    * True if this flow was truncated due to a timeout or other flush
    * of the sensor's cache. There may be a matching flow record with
    * the continuation flag set.
    * @group Flags
    */
  def truncated: Boolean = (toByte & TCPState.BIT_TRUNCATED) != 0
  /**
    * True if this flow carries on after truncation due to a timeout
    * or other flush of the sensor's cache. There may be a matching
    * flow record with the truncated flag set.
    * @group Flags
    */
  def continuation: Boolean = (toByte & TCPState.BIT_CONTINUATION) != 0
  /**
    * True if this flow's IP addresses are IPv6 addresses.
    * @group Flags
    */
  def isIPv6: Boolean = (toByte & TCPState.BIT_IS_IPV6) != 0

  /**
    * The bitwise NOT of a set of TCP state flags.
    */
  def unary_~ : TCPState =
    TCPState(((~toByte) & TCPState.BITS_ALL_FLAGS).toByte)

  /**
    * The bitwise AND of two TCP state flag sets.
    */
  def &(o: TCPState): TCPState = TCPState((toByte | o.toByte).toByte)

  /**
    * The bitwise OR of two TCP state flag sets.
    */
  def |(o: TCPState): TCPState = TCPState((toByte & o.toByte).toByte)

  /**
    * The bitwise XOR of two TCP state flag sets.
    */
  def ^(o: TCPState): TCPState = TCPState((toByte ^ o.toByte).toByte)

  /**
    * Returns a string representation of this TCPState value using the
    * following characters:
    *
    *   - `T` truncated
    *   - `C` continuation
    *   - `F` finFollowed
    *   - `S` uniformPacketSize
    *   - `x` expandedFlags
    *   - `6` isIPv6
    */
  override def toString(): String = TCPState.attributeChars.map({
    case (b, c) => if ( (toByte & b) != 0 ) c else " "
  }).mkString
}

object TCPState {
  private val BIT_EXPANDED_FLAGS: Byte      = 0x01
  private val BIT_FIN_FOLLOWED: Byte        = 0x08
  private val BIT_UNIFORM_PACKET_SIZE: Byte = 0x10
  private val BIT_TRUNCATED: Byte           = 0x20
  private val BIT_CONTINUATION: Byte        = 0x40
  private val BIT_IS_IPV6: Byte             = 0x80.toByte

  private val BITS_ALL_FLAGS: Byte =
    (BIT_EXPANDED_FLAGS | BIT_FIN_FOLLOWED | BIT_UNIFORM_PACKET_SIZE |
      BIT_TRUNCATED | BIT_CONTINUATION | BIT_IS_IPV6).toByte

  private val attributeChars = Seq(
    BIT_TRUNCATED -> 'T',
    BIT_CONTINUATION -> 'C',
    BIT_FIN_FOLLOWED -> 'F',
    BIT_UNIFORM_PACKET_SIZE -> 'S',
    BIT_EXPANDED_FLAGS -> 'x',
    BIT_IS_IPV6 -> '6'
  )

  /**
    * Given an integer, returns the TCPState value represented by the
    * low 8 bits of that integer.
    */
  def apply(intVal: Int): TCPState = TCPState(intVal.toByte)

  /**
    * Returns a TCPState value constructed from a string expressing
    * the state using the following characters:
    *
    *   - `T` truncated
    *   - `C` continuation
    *   - `F` finFollowed
    *   - `S` uniformPacketSize
    *   - `x` expandedFlags
    *   - `6` isIPv6
    *
    * Other characters are ignored.
    */
  def ofString(s: String): TCPState = TCPState(
    TCPState.attributeChars.map({ case (b, c) =>
      if ( s contains c ) b else 0
    }).reduce((_ | _)).toByte
  )
}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
