// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.silk

import java.io.{Reader => JReader}
import java.time.{Instant, OffsetDateTime, ZoneOffset}
import java.time.format.DateTimeParseException
import scala.util.Try
import scala.util.parsing.input.Reader
import config.{
  ClassConfig, FlowTypeConfig, FlowTypeMap, GroupConfig, SensorConfig,
  SensorMap, SilkConfigParser}

/**
  * SiLK data spool configuration.
  *
  * @param version The version of the config file format used.
  * @param defaultClassName The default class to be examined if none
  *     is specified.
  * @param packingLogicPath The path to the plugin to be loaded by the
  *     packer for determining where to pack flows.
  * @param pathFormat The format used for filenames in the data spool.
  * @param groups The sensor groups defined in this configuration.
  * @param sensors The sensors defined in this configuration, usable
  *     as a value of type `Map[Sensor, SensorConfig]`.
  * @param classes The classes defined in this configuration.
  */
case class SilkConfig(
  version: Option[Int],
  defaultClassName: Option[String],
  packingLogicPath: Option[String],
  pathFormat: String,
  groups: Map[String, GroupConfig],
  sensors: SensorMap,
  classes: Map[String, ClassConfig]
) {

  /** The flowtypes defined in this configuration in any class, usable
    * as a value of type `Map[FlowType, FlowTypeConfig]`. */
  def flowTypes: FlowTypeMap = classes.values.flatMap(_.flowTypes).toMap

  /** Returns true if the config version supports sensor descriptions. */
  def supportsSensorDescriptions: Boolean = version match {
    case Some(v) if v < 2 => false
    case _ => true
  }

  /** A string using shell glob syntax which matches all data files for
    * this config.
    */
  def globAll: String = globAll(pathFormat)

  private[this] def globAll(format: String): String = {
    def formatGlobChar(c: Char): String = c match {
      case 'C' | 'F' | 'N' | 'T' | 'f' | 'n' => "*"
      case 'Y' => "????"
      case 'm' | 'd' | 'H' => "??"
      case 'x' => globAll("%F-%N_%Y%m%d.%H")
      case other => other.toString
    }
    val parts = format.split('%')
    (parts.head +: (for {
      part <- parts.tail
      globChar = part.head
      text = part.tail
      resultPart <- Seq(formatGlobChar(globChar), text)
    } yield resultPart)).mkString("").replaceAll("\\*+", "*")
  }

  private val filePattern = """([^ /]+-[^ _/]+)_(\d\d\d\d)(\d\d)(\d\d)\.(\d\d)(?:\..*)?""".r
  def filenameToGlobInfo(path: String): Option[(Instant, FlowType, Sensor)] = {
    // %F-%N_%Y%m%d.%H
    // Flowtype name, sensor name, year, month, day, and hour
    val filename = path.split('/').last
    filename match {
      case filePattern(flowTypeAndSensor, year, month, day, hour) => {
        val candidates = for {
          flowTypeConf <- flowTypes.values
          flowTypeName = flowTypeConf.flowTypeName
          if flowTypeAndSensor.startsWith(flowTypeName + "-")
          sensorName = flowTypeAndSensor.slice(
            flowTypeName.length + 1, flowTypeAndSensor.length)
          if sensors.byName.contains(sensorName)
          sensorConf = sensors.byName(sensorName)
        } yield (flowTypeConf.id, sensorConf.id)
        candidates.headOption match {
          case Some((flowTypeId, sensorId)) =>
            try {
              Some((Instant.parse(s"${year}-${month}-${day}T${hour}:00:00Z"),
                    flowTypeId, sensorId))
            } catch {
              case _: DateTimeParseException => None
            }
          case _ => None
        }
      }
      case _ => None
    }
  }

  /** Given a tuple containing a SiLK Record's starting time, FlowType,
    * and Sensor, return a partial path, relative to the root of the
    * SiLK data repository, to the hourly file holding that record. */
  def fileInfoToPath(t: (Instant, FlowType, Sensor)): String =
    fileInfoToPath(pathFormat,
      (t._1.atOffset(ZoneOffset.UTC), flowTypes(t._2), sensors(t._3)))

  private[this] def fileInfoToPath(
    format: String, t: (OffsetDateTime, FlowTypeConfig, SensorConfig)
  ): String = {

    def expandPathChar(c: Char): String = c match {
      case 'H' => f"${t._1.getHour()}%02d"
      case 'Y' => f"${t._1.getYear()}%04d"
      case 'd' => f"${t._1.getDayOfMonth()}%02d"
      case 'm' => f"${t._1.getMonthValue()}%02d"

      case 'C' => t._2.className
      case 'F' => t._2.flowTypeName
      case 'T' => t._2.typeName
      case 'f' => t._2.id.toShort.toString

      case 'N' => t._3.name
      case 'n' => t._3.id.toShort.toString

      case 'x' => fileInfoToPath("%F-%N_%Y%m%d.%H", t)

      case other => other.toString
    }

    val parts = format.split('%')
    (parts.head +: (for {
      part <- parts.tail
      pathChar = part.head
      text = part.tail
      resultPart <- Seq(expandPathChar(pathChar), text)
    } yield resultPart)).mkString("")
  }

  // The following methods are for the use of the parser to ease
  // step-wise construction of a complete SiLK configuration. These do
  // not do any checking for conflicts--that's handled in the parser
  // so errors can be reported.

  /** Tries to return a set of sensor names given a sensor or
    * group reference. */
  private[silk] def expandSensorSpecs(specs: Iterable[String]): Try[Iterable[String]] = Try {
    specs.flatMap({
      case groupRef if groupRef.startsWith("@") => {
        val groupName = groupRef.substring(1)
        groups.getOrElse(groupName, {
          throw new Error(s"group '${groupName}' is not defined")
        }).sensorNames
      }
      case sensorName => {
        if ( !sensors.byName.contains(sensorName) ) {
          throw new Error(s"sensor '${sensorName}' is not defined")
        } else {
          Iterator(sensorName)
        }
      }
    })
  }

  /**
    * Returns the configuration with an empty class entry for the
    * given class, if it's not already present. Otherwise returns the
    * configuration unchanged.
    */
  private[silk] def withClass(className: String): SilkConfig =
    if ( classes.contains(className) ) this else {
      copy(classes = classes + (className ->
        ClassConfig(name = className, defaultTypeNames = Set.empty,
          sensorNames = Set.empty,
          flowTypes = Map.empty[FlowType, FlowTypeConfig])))
    }

  /**
    * Returns the configuration with additional default types for the
    * named class.
    */
  private[silk] def withClassDefaultTypes(
    className: String, typeNames: Iterable[String]): SilkConfig =
  {
    val classConf = classes(className)
    copy(classes = classes + (className ->
      classConf.copy(
        defaultTypeNames = classConf.defaultTypeNames ++ typeNames)))
  }

  /**
    * Returns the configuration with additional sensors for the named
    * class.
    */
  private[silk] def withClassSensors(
    className: String, sensorNames: Iterable[String]): SilkConfig =
  {
    val classConf = classes(className)
    copy(classes = classes + (className ->
      classConf.copy(
        sensorNames = classConf.sensorNames ++ sensorNames)))
  }

  /**
    * Returns the configuration with an additional flowtype entry.
    */
  private[silk] def withFlowType(
    id: FlowType, className: String, typeName: String,
    flowTypeName: String): SilkConfig =
  {
    val classConf = classes(className)
    copy(classes = classes + (className ->
      classConf.copy(
        flowTypes = classConf.flowTypes.byId + (id ->
          FlowTypeConfig(id, className, typeName, flowTypeName)))))
  }

  /**
    * Returns the configuration with the default class name set.
    */
  private[silk] def withDefaultClassName(className: String): SilkConfig =
    copy(defaultClassName = Some(className))

  /**
    * Returns the configuration with an empty group entry for the
    * given name, if needed. Otherwise returns the configuration
    * unchanged.
    */
  private[silk] def withGroup(groupName: String): SilkConfig =
    if ( groups.contains(groupName) ) this else {
      copy(groups = groups + (groupName ->
        GroupConfig(name = groupName, sensorNames = Set.empty)))
    }

  /**
    * Returns the configuration with additional sensors for the named
    * group.
    */
  private[silk] def withGroupSensors(
    groupName: String, sensorNames: Iterable[String]): SilkConfig =
  {
    val groupConf = groups(groupName)
    copy(groups = groups + (groupName ->
      groupConf.copy(
        sensorNames = groupConf.sensorNames ++ sensorNames)))
  }

  /**
    * Returns the configuration with the packing logic path set.
    */
  private[silk] def withPackingLogicPath(path: String): SilkConfig =
    copy(packingLogicPath = Some(path))

  /**
    * Returns the configuration with the path format set.
    */
  private[silk] def withPathFormat(format: String): SilkConfig =
    copy(pathFormat = format)

  /**
    * Returns the configuration with an additional sensor entry.
    */
  private[silk] def withSensor(
    id: Sensor, name: String, description: Option[String]): SilkConfig =
  {
    copy(sensors = sensors.byId + (id ->
      SensorConfig(id, name, description)))
  }

  /**
    * Returns the configuration with the configuration format version set.
    */
  private[silk] def withVersion(versionNum: Int): SilkConfig =
    copy(version = Some(versionNum))

}

object SilkConfig {

  private val defaultPathFormat = "%T/%Y/%m/%d/%x"
  private val baseConf = SilkConfig(
    version = None, defaultClassName = None, packingLogicPath = None,
    pathFormat = defaultPathFormat, groups = Map.empty,
    sensors = Map.empty[Sensor, SensorConfig], classes = Map.empty)

  def apply(in: CharSequence): SilkConfig =
    SilkConfigParser.parseAll(SilkConfigParser.top(baseConf), in).get
  def apply(in: JReader): SilkConfig =
    SilkConfigParser.parseAll(SilkConfigParser.top(baseConf), in).get
  def apply(in: Reader[Char]): SilkConfig =
    SilkConfigParser.parseAll(SilkConfigParser.top(baseConf), in).get

}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
