// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.silk
package io

import org.apache.hadoop.conf.Configuration
import org.apache.hadoop.io.compress.CompressionCodecFactory
import java.io.{DataInputStream, InputStream}
import java.util.zip.Inflater

/**
  * A [[BufferReader]] which reads zlib-compressed data from SiLK file
  * streams, using either Hadoop or Java compressor codecs.
  */
private[silk] class ZlibInputStreamBuffer(
  configuration: Option[Configuration], inStream: InputStream)
    extends BufferReader
{
  private trait Decompressor {
    def setInput(b: Array[Byte], off: Int, len: Int): Unit
    def needsInput(): Boolean
    def decompress(b: Array[Byte], off: Int, len: Int): Int
    def reset(): Unit
  }

  private def hadoopDecompressor: Option[Decompressor] = try {
    for {
      conf <- configuration
      c <- Option(new CompressionCodecFactory(conf).getCodecByName("gzip"))
    } yield new Decompressor {
      val decomp = c.createDecompressor()
      def setInput(b: Array[Byte], off: Int, len: Int): Unit =
        decomp.setInput(b, off, len)
      def needsInput(): Boolean = decomp.needsInput()
      def decompress(b: Array[Byte], off: Int, len: Int): Int =
        decomp.decompress(b, off, len)
      def reset(): Unit = decomp.reset()
    }
  } catch {
    case _: Exception => None
  }

  private def javaDecompressor: Decompressor = new Decompressor {
    val inflater = new Inflater()
    def setInput(b: Array[Byte], off: Int, len: Int): Unit =
      inflater.setInput(b, off, len)
    def needsInput(): Boolean = inflater.needsInput()
    def decompress(b: Array[Byte], off: Int, len: Int): Int =
      inflater.inflate(b, off, len)
    def reset(): Unit = inflater.reset()
  }

  private val in: DataInputStream = inStream match {
    case in: DataInputStream => in
    case _ => new DataInputStream(inStream)
  }

  private val decompressor: Decompressor =
    hadoopDecompressor.getOrElse(javaDecompressor)

  def close(): Unit = {
    in.close()
  }
  def getNextBuffer(): (Array[Byte], Int) = {
    val compressedSize = in.readInt()
    val uncompressedSize = in.readInt()
    val compressedData = Array.ofDim[Byte](compressedSize)
    val uncompressedData = Array.ofDim[Byte](uncompressedSize)
    in.readFully(compressedData)
    decompressor.setInput(compressedData, 0, compressedSize)
    val uncompressedSizeResult =
      decompressor.decompress(uncompressedData, 0, uncompressedSize)
    assert(decompressor.needsInput)
    assert(uncompressedSizeResult == uncompressedSize)
    decompressor.reset()
    (uncompressedData, uncompressedSize)
  }
}

private[silk] object ZlibInputStreamBuffer {
  def apply(conf: Configuration, inStream: InputStream): ZlibInputStreamBuffer =
    new ZlibInputStreamBuffer(Option(conf), inStream)
  def apply(inStream: InputStream): ZlibInputStreamBuffer =
    new ZlibInputStreamBuffer(None, inStream)
}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
