// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.silk
package io
package prefixmap

import scala.collection.immutable.Vector

import org.cert.netsa.io.silk.PrefixMapProtocolPortPair
import org.cert.netsa.data.net.{Protocol, Port}


private[silk] class PrefixMapReaderProtoPort (
  reader: BufferReader,
  header: Header
) extends PrefixMapReader[PrefixMapProtocolPortPair](reader, header)
{
  // The prefix map as an array of Integers
  private[this] val tree: Array[Int] = readTree()

  // The dictionary of words
  private[this] val dictionary: Array[String] = readDictionary()

  // a variable to trace our path into the tree
  private[this] var path: Vector[Int] = Vector(0)

  private[this] var nextLeaf: Int = 0

  private[this] var nextRange = findRange()

  /*
   * Use the existing values in 'path' to determine the current
   * protocol-port pair; these represent the start of the range.
   * Then, starting from the final index in 'path', travel down the
   * left hand side of each node in the tree until a leaf is reached.
   *
   * Update the class value 'nextLeaf' with that leaf's value.
   *
   * Use the depth of the tree to determine the ending protocol-port
   * pair.
   *
   * Update the class value 'path' with the current path.
   */
  private[this] def findRange(): PrefixMapProtocolPortPair = {
    var start: Int = 0

    // compute the lower proto-port pair by traversing the path
    val it = path.iterator
    var shift = 23
    while ( it.hasNext ) {
      start = start | ((it.next() & 0x1) << shift)
      shift = shift - 1
    }

    // travel down left hand side until we reach a leaf
    var idx = tree(path.last)
    while ( !isLeaf(idx) ) {
      path = path :+ idx
      idx = tree(idx)
    }
    nextLeaf = idx & 0x7fffffff

    val end: Int = start | (0xffffff >>> path.length)

    //print("leaf = " + nextLeaf
    //  + "; start = " + ((start >>> 16) & 0xff) + "/" + (start & 0xffff)
    //  + "; end = " +  ((end >>> 16) & 0xff) + "/" + (end & 0xffff)
    //  + "; depth = " + path.length + "; path =")
    //for (x <- path) { print(" " + x) }
    //println("")

    PrefixMapProtocolPortPair(
      Protocol(start >>> 16), Port(start & 0xffff),
      Protocol((end >>> 16) & 0xff), Port(end & 0xffff))
  }

  /*
   * Strip the trailing right-hand branches from the current values in
   * 'path', move the bottom most left-hand node to use its right-hand
   * branch, and call findRange() to find the left-most leaf.
   *
   * When there are no more entries, sets the class value 'nextLeaf'
   * to -1 and returns the class's existing 'nextRange' value.
   */
  private[this] def getNextRange(): PrefixMapProtocolPortPair = {
    // find bottom-most node whose right branch has not been visited
    val n = path.lastIndexWhere(isEven)
    if (-1 == n) {
      // there is no more data; set nextLeaf to -1 and return the
      // existing value of nextRange
      if ( !path.isEmpty ) {
        path = Vector.empty
      }
      nextLeaf = -1
      return nextRange
    }

    // remove the bottom-most right branches from path, then replace
    // the bottom-most left value with its right branch
    path = path.take(n) :+ (path(n) + 1)

    findRange()
  }

  override def hasNext: Boolean = {
    -1 != nextLeaf
  }

  override def next(): (PrefixMapProtocolPortPair, String) = {
    if (-1 == nextLeaf) {
      throw new NoSuchElementException("PrefixMapReader")
    }

    val start = nextRange
    val leaf = nextLeaf

    var end = start
    nextRange = getNextRange()
    while ( leaf == nextLeaf ) {
      end = nextRange
      nextRange = getNextRange()
    }

    (PrefixMapProtocolPortPair(start.startProtocol, start.startPort, end.endProtocol, end.endPort), dictionary(leaf))
  }
}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
