// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.silk
package io

import java.io.{DataOutputStream, OutputStream}
import org.apache.hadoop.conf.Configuration
import org.apache.hadoop.io.compress.{CompressionCodecFactory, Compressor}


/**
  * A [[BufferWriter]] which compresses data using LZO and writes it
  * to a stream in a form compatible with SiLK file streams, using
  * Hadoop compression codecs.
  *
  * @see [[LzoOutputStreamBuffer$ the companion object]] for more
  * details.
  */
private[silk] class LzoOutputStreamBuffer(
  configuration: Configuration,
  outStream:     OutputStream)
    extends BufferWriter
{
  // The output stream
  private val out: DataOutputStream = outStream match {
    case out: DataOutputStream => out
    case _ => new DataOutputStream(outStream)
  }

  // The compression engine
  private val compressor: Compressor = try {
    val cf = new CompressionCodecFactory(configuration)
    val c = cf.getCodecByName("lzo")
    c.createCompressor()
  } catch {
    case ex: Exception =>
      throw new RuntimeException(
        "Unable to create lzo compressor--is lzo properly configured?", ex)
  }

  /**
    * Return the size of a buffer that is guaranteed to be large
    * enough to hold the result of compressing `sourceLen` bytes of
    * data.
    */
  private[this] def compressBound(sourceLen: Int): Int = {
    // Formula taken from the LZO FAQ:
    // http://www.oberhumer.com/opensource/lzo/lzofaq.php
    sourceLen + (sourceLen >>> 4) + 64 + 3
  }

  /**
    * Compress and write any pending data, finalize the compression
    * engine, and flush the output stream.
    */
  def end(): Unit = {
    assert(compressor.needsInput)
    compressor.end()
    out.flush()
  }

  /**
    * Write the first `length` bytes of data in `buffer` to the output
    * stream.
    *
    * @throws java.io.IOException when an error occurs.
    */
  def putBuffer(uncompressedData: Array[Byte], uncompressedSize: Int): Unit = {
    if (0 == uncompressedSize) {
      return
    }

    // create buffer to hold the result.  FIXME: Consider moving this
    // array outside the function
    val maxCompressedSize = compressBound(uncompressedSize)
    val compressedData = Array.ofDim[Byte](maxCompressedSize)

    // compress it
    assert(compressor.needsInput)
    compressor.setInput(uncompressedData, 0, uncompressedSize)
    // the other compression files have a call to compressor.finish()
    // here, but the LZO compression works without it

    val compressedSize = compressor.compress(compressedData, 0, maxCompressedSize)

    // write the two sizes and the compressed block
    out.writeInt(compressedSize)
    out.writeInt(uncompressedSize)
    out.write(compressedData, 0, compressedSize)

    /*
     * Comment out this block since compressor.finish() is not used
     * above.

    // the following block should not be necessary but it is here
    // "just in case"
    while ( !compressor.finished() ) {
      val size = compressor.compress(compressedData, 0, maxCompressedSize)
      //println("Writing an additional block of " + size + " bytes")
      out.write(compressedData, 0, size)
    }
     */

    assert(compressor.needsInput)
    assert(0 == compressor.compress(compressedData, 0, maxCompressedSize))
    compressor.reset()
  }
}


/**
  * The LzoOutputStreamBuffer object provides support for creating an
  * [[LzoOutputStreamBuffer]].
  */
private[silk] object LzoOutputStreamBuffer {
  def apply(conf: Configuration, outStream: OutputStream):
      LzoOutputStreamBuffer =
  {
    new LzoOutputStreamBuffer(conf, outStream)
  }
}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
