// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.silk
package io
package unpackers

import org.cert.netsa.data.net.{
  IPv4Address, Port, Protocol, SNMPInterface, TCPFlags}

import BufferUtil._

/*
**  RWWWW VERSION 1
**  RWWWW VERSION 2
**
**    uint32_t      sIP;             //  0- 3  Source IP
**    uint32_t      dIP;             //  4- 7  Destination IP
**
**    uint32_t      pef;             //  8-11
**    // uint32_t     pkts      :20; //        Count of packets
**    // uint32_t     elapsed   :11; //        Duration of flow
**    // uint32_t     pflag     : 1; //        'pkts' requires multiplier?
**
**    uint32_t      sbb;             // 12-15
**    // uint32_t     sTime     :12; //        Start time--offset from hour
**    // uint32_t     bPPkt     :14; //        Whole bytes-per-packet
**    // uint32_t     bPPFrac   : 6; //        Fractional bytes-per-packet
**
**    uint16_t      clnt_port;       // 16-17  Non-Web Port
**
**    uint8_t       wrf;             // 18
**    // uint8_t      srcIsSrv  : 1; //        1 if srvPort is src; 0 if dest
**    // uint8_t      pad       : 1; //        padding/reserved
**    // uint8_t      flags     : 6; //        OR of all TCP flags on all pkts
**
**    uint8_t       wPort;           // 19
**    // uint8_t      srvPort   : 2; //        server port: 0=80; 1=443; 2=8080
**    // uint8_t      pad       : 6; //        padding/reserved
**
**
**  20 bytes on disk.
*/

private[silk] object FT_RWWWW_v1 extends Unpacker {
  val recordLength = 20
  def unpack(
    buffer: Array[Byte], offset: Int, swap: Boolean, header: Header): RWRec =
  {
    val sIP = getIPv4Address(buffer, offset, swap)
    val dIP = getIPv4Address(buffer, offset + 4, swap)
    val (startTime, elapsed, bytes, packets) =
      unpackSbbPef(buffer, offset + 12, offset + 8, swap, header)
    val clientPort = getPort(buffer, offset + 16, swap)
    val wrf = getInt8(buffer, offset + 18, swap)
    val wPort = getInt8(buffer, offset + 19, swap)

    val sourceIsServer = (wrf & 0x80) != 0
    val serverPort = decodeWWWPort((wPort >> 6) & 0x03)

    val flags = TCPFlags((wrf & 0x3F).toByte)
    val (sPort, dPort) = if ( sourceIsServer ) {
      (serverPort, clientPort)
    } else {
      (clientPort, serverPort)
    }
    val protocol = Protocol.TCP

    // unsupplied
    val flowType = header.packedFlowTypeId
    val sensor = header.packedSensorId
    val input = SNMPInterface(0)
    val output = SNMPInterface(0)
    val initFlags = TCPFlags(0)
    val restFlags = TCPFlags(0)
    val tcpState = TCPState(0)
    val application = Port(0)
    val memo: Short = 0
    val nhIP = IPv4Address(0)
    RWRec(
      startTime, elapsed, sPort, dPort, protocol, flowType, sensor,
      flags, initFlags, restFlags, tcpState, application, memo, input,
      output, packets, bytes, sIP, dIP, nhIP).maybeClearTCPStateExpanded
  }
}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
