// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.silk
package io
package prefixmap

import scala.collection.immutable.Vector

import org.cert.netsa.data.net.{
  IPv6Address, IPBlock, IPv6Block}


private[silk] class PrefixMapReaderIPv6 (
  reader: BufferReader,
  header: Header
) extends PrefixMapReader[IPBlock](reader, header)
{
  // The prefix map as an array of Integers
  private[this] val tree: Array[Int] = readTree()

  // The dictionary of words (empty array for country-code pmap)
  private[this] val dictionary: Array[String] = readDictionary()

  // is this the first time next() is being called?
  private[this] var firstCall = true

  // a variable to trace our path into the tree; initialize it with
  // the left hand side of the root node
  private[this] var path: Vector[Int] = Vector(0)

  override def hasNext: Boolean = {
    path.lastIndexWhere(isEven) != -1
  }

  override def next(): (IPBlock, String) = {
    // the IP address to return
    val ip: Array[Short] = Array.empty.padTo(8, 0.toShort)

    if (firstCall) {
      // this is the first call to next(); no need to do anything
      firstCall = false
    } else {
      // find bottom-most node whose right branch has not been visited
      val n = path.lastIndexWhere(isEven)
      if (-1 == n) {
        if ( !path.isEmpty ) {
          path = Vector.empty
        }
        throw new NoSuchElementException("PrefixMapReader")
      }

      // remove the bottom-most right branches from path, then replace
      // the bottom-most left value with its right branch
      path = path.take(n) :+ (path(n) + 1)

      // create the current IP address by traversing the path
      val it = path.iterator
      var pos = 0
      var shift = 15
      while ( it.hasNext ) {
        ip(pos) = (ip(pos) | ((it.next() & 0x1) << shift)).toShort
        if ( shift > 0 ) {
          shift = shift - 1
        } else {
          pos = pos + 1
          shift = 15
        }
      }
    }

    // travel down left hand side until we reach a leaf
    var idx = tree(path.last)
    while ( !isLeaf(idx) ) {
      path = path :+ idx
      idx = tree(idx)
    }

    // create the IP block to return
    val block = IPv6Block(IPv6Address(ip), path.length)

    if (hasDictionary) {
      return (block, dictionary(idx & leafValueMask))
    }
    return (block, getCountryCode(idx))
  }
}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
