// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.silk
package io
package unpackers

import org.cert.netsa.data.net.{IPv4Address, Port, TCPFlags}

import BufferUtil._

/*
**  FLOWCAP VERSION 2
**
**    uint32_t      sIP;             //  0- 3  Source IP
**    uint32_t      dIP;             //  4- 7  Destination IP
**
**    uint32_t      bytes;           //  8-11  Byte count
**
**    uint32_t      sTime;           // 12-15  Start time as UNIX epoch secs
**
**    uint16_t      elapsed;         // 16-17  Duration of flow in seconds
**    uint16_t      sPort;           // 18-19  Source port
**
**    uint16_t      dPort;           // 20-21  Destination port
**    uint8_t       input;           // 22     SNMP Input
**    uint8_t       output;          // 23     SNMP Output
**
**    uint8_t       pkts[3]          // 24-26  Count of packets
**    uint8_t       proto            // 27     Protocol
**
**    uint8_t       flags            // 28     OR of TCP Flags on all pkts
**    uint8_t       first_flags;     // 29     TOS (ignored)
**
**
**  30 bytes on disk.
*/

private[silk] object FT_FLOWCAP_v2 extends Unpacker {
  val recordLength = 30
  def unpack(
    buffer: Array[Byte], offset: Int, swap: Boolean, header: Header): RWRec =
  {
    val sIP = getIPv4Address(buffer, offset, swap)
    val dIP = getIPv4Address(buffer, offset + 4, swap)
    val bytes = getBytes32(buffer, offset + 8, swap)
    val startTime = getStartTimeSecs(buffer, offset + 12, swap)
    val elapsed = getElapsedSecs16(buffer, offset + 16, swap)
    val sPort = getPort(buffer, offset + 18, swap)
    val dPort = getPort(buffer, offset + 20, swap)
    val input = getSNMPInterface8(buffer, offset + 22, swap)
    val output = getSNMPInterface8(buffer, offset + 23, swap)
    val packets = getPackets24(buffer, offset + 24, swap)
    val protocol = getProtocol(buffer, offset + 27, swap)
    val flags = getTCPFlags(buffer, offset + 28, swap)
    // unsupplied
    val flowType = FlowType(0)
    val sensor = Sensor(0)
    val initFlags = TCPFlags(0)
    val restFlags = TCPFlags(0)
    val tcpState = TCPState(0)
    val application = Port(0)
    val memo: Short = 0
    val nhIP = IPv4Address(0)
    RWRec(
      startTime, elapsed, sPort, dPort, protocol, flowType, sensor,
      flags, initFlags, restFlags, tcpState, application, memo, input,
      output, packets, bytes, sIP, dIP, nhIP).maybeClearTCPStateExpanded
  }
}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
