// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.silk

import java.time.{Duration, Instant}
import org.cert.netsa.data.net.{
  ICMPCode, ICMPType, IPAddress, Port, Protocol, SNMPInterface, TCPFlags}

// Could make some of these Options, based on not all record types
// having them, and also based on not all records having them
// (i.e. icmpType and icmpCode are meaningful for ICMP but not TCP,
// flags &c only useful for TCP, etc.

/**
  * A SiLK flow record.
  *
  * Note that in addition to the fields of the case class, some
  * derived fields are also provided. (See below.)
  *
  * @param startTime The instant that the first packet in this flow
  *     was observed.
  * @param elapsed The duration between the instants the first and
  *     last packets in this flow were observed.
  * @param sPort The source port of this flow, or zero if this flow is
  *     neither a TCP nor a UDP flow.
  * @param dPort The destination port of this flow, or zero if this flow
  *     is neither a TCP nor a UDP flow.
  * @param protocol The IP protocol of this flow.
  * @param flowType The SiLK flow type (class and type) of this flow,
  *     or `FlowType(0)` if unknown.
  * @param sensor The SiLK sensor that observed this flow, or `Sensor(0)`
  *     if unknown.
  * @param flags The union of all TCP flags observed in this flow, or
  *     `TCPFlags(0)` if this flow is not a TCP flow.
  * @param initFlags The flags observed in the initial packet of this
  *     TCP flow, or `TCPFlags(0)` if this flow is not a TCP flow or
  *     if extended flags are not available.
  * @param restFlags The union of all TCP flags observed after the
  *     initial packet of this flow, or `TCPFlags(0)` if this flow is
  *     not a TCP flow or if extended flags are not available.
  * @param tcpState Flags relating to the observed status of this
  *     flow, including whether extended TCP flags are available. See
  *     [[TCPState]] for more details.
  * @param application The detected application of this flow,
  *     expressed as the common port number for that application, or
  *     `Port(0)` if no application was detected.
  * @param memo A [[scala.Short Short]] value stored as a memo on this
  *     flow, or zero if no such memo has been set.
  * @param input The input SNMP routing interface for this flow, or
  *     `SNMPInterface(0)` if routing information is not available.
  * @param output The output SNMP routing interface for this flow, or
  *     `SNMPInterface(0)` if routing information is not available.
  * @param packets The number of IP packets observed in this flow.
  * @param bytes The number of bytes in packets observed in this flow.
  * @param sIP The source IP address of packets in this flow.
  * @param dIP The destination IP address of packets in this flow.
  * @param nhIP The next-hop IP address of packets in this flow, or
  *     `IPAddress("0.0.0.0")` or `IPAddress("::")` if routing
  *     information is not available.
  */
case class RWRec(
  val startTime: Instant,
  val elapsed: Duration,
  val sPort: Port,
  val dPort: Port,
  val protocol: Protocol,
  val flowType: FlowType,
  val sensor: Sensor,
  val flags: TCPFlags,
  val initFlags: TCPFlags,
  val restFlags: TCPFlags,
  val tcpState: TCPState,
  val application: Port,
  val memo: Short,
  val input: SNMPInterface,
  val output: SNMPInterface,
  val packets: Long,
  val bytes: Long,
  val sIP: IPAddress,
  val dIP: IPAddress,
  val nhIP: IPAddress
) {
  /**
    * The instant that the last packet in this flow was observed.
    * @group Derived Members
    */
  def endTime: Instant = startTime.plus(elapsed)
  /**
    * The type of this ICMP flow, or garbage if this is a TCP or UDP
    * flow, or `ICMPType(0)` if this is not an ICMP, TCP, or UDP flow.
    * @group Derived Members
    */
  def icmpType: ICMPType = ICMPType((dPort.toShort >> 8).toByte)
  /**
    * The code of this ICMP flow, or garbage if this is a TCP or UDP
    * flow, or `ICMPType(0)` if this is not an ICMP, TCP, or UDP flow.
    * @group Derived Members
    */
  def icmpCode: ICMPCode = ICMPCode(dPort.toShort.toByte)
  /**
    * True if this flow's addresses are IPv6 addresses.
    * @group Derived Members
    */
  def isIPv6: Boolean = tcpState.isIPv6
}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
