// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.silk

import java.io.{EOFException, InputStream}

import io.{BufferReader, LzoInputStreamBuffer, RawInputStreamBuffer,
  SnappyInputStreamBuffer, Unpacker, ZlibInputStreamBuffer}

/**
  * A reader of SiLK flow records. This is usable as an
  * [[scala.Iterator Iterator]] over [[RWRec]] records.
  *
  * See [[RWRecReader$ the companion object]] for more details.
  */
class RWRecReader private (
  private[this] val reader: BufferReader,
  private[this] val unpacker: Unpacker,
  val header: Header
) extends Iterator[RWRec] {

  /**
    * Current buffer of bytes being processed.
    */
  private[this] var buffer = Array.empty[Byte]
  /**
    * Length of current buffer in bytes.
    */
  private[this] var bufferLength = 0
  /**
    * Current offset into current buffer.
    */
  private[this] var offset = 0
  /**
    * True if reading of the input has completed, and no more records
    * should be read or produced.
    */
  private[this] var done = false
  /**
    * The length in bytes of records being processed by this reader.
    */
  private[this] val recordLength = unpacker.recordLength

  /**
    * Read a new buffer into memory if needed. Does no work if the
    * current buffer still has unread records, or if the reader is
    * done. Sets done to true if the end of file is reached.
    */
  private[this] def fillBuffer(): Unit = {
    if ( ! done ) {
      if ( offset + recordLength > bufferLength ) {
        try {
          val (newBuffer, newLength) = reader.getNextBuffer()
          buffer = newBuffer
          bufferLength = newLength
          offset = 0
        } catch {
          case _: EOFException => {
            reader.close()
            buffer = Array.empty
            done = true
          }
        }
      }
    }
  }

  override def hasNext: Boolean = {
    fillBuffer()
    !done
  }

  override def next(): RWRec = {
    fillBuffer()
    if ( done ) {
      throw new NoSuchElementException("RWRecReader")
    }
    val rec = unpacker.unpack(buffer, offset, !header.isBigEndian, header)
    offset += recordLength
    rec
  }

}

object RWRecReader {

  /**
    * Creates and returns a reader from the provided input
    * stream. Does not support compressed data.
    *
    * @throws SilkDataFormatException if the input stream is malformed
    *     or uses compression.
    */
  def ofInputStream(s: InputStream): RWRecReader = {
    val header = Header.readFrom(s)
    val unpacker =
      FileFormat.unpacker(header.fileFormat, header.recordVersion) match {
        case Some(unpacker) => unpacker
        case None =>
          throw new SilkDataFormatException(
            "Unrecognized file format and version")
      }
    val bufferSize = (65536 / unpacker.recordLength) * unpacker.recordLength
    val bufferReader = header.compressionMethod match {
      case CompressionMethod.NONE => RawInputStreamBuffer(s, bufferSize)
      case CompressionMethod.ZLIB => ZlibInputStreamBuffer(s)
      case CompressionMethod.LZO1X => LzoInputStreamBuffer(s)
      case CompressionMethod.SNAPPY => SnappyInputStreamBuffer(s)
      case _ => throw new SilkDataFormatException("Unsupported compression method (without Hadoop)")
    }

    new RWRecReader(bufferReader, unpacker, header)
  }

}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
