// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.silk
package io

import com.github.ghik.silencer.silent
import java.lang.{Byte => JByte, Integer => JInt, Short => JShort}
import java.time.{Duration, Instant}
import org.cert.netsa.data.net.{IPAddress, IPv4Address, IPv6Address,
  Port, Protocol, SNMPInterface, TCPFlags}

/**
  * A set of utility routines for working with binary input stored in
  * an array of bytes. Does not aggressively check array boundaries,
  * so reading near the end will result in an
  * [[java.lang.ArrayIndexOutOfBoundsException ArrayIndexOutOfBoundsException]].
  */
private[silk] object BufferUtil {

  /**
    * Fetches an 8-bit integer value from the given offset.
    */
  @silent("swap .* never used")
  def getInt8(buffer: Array[Byte], offset: Int, swap: Boolean): Byte =
    buffer(offset + 0)

  /**
    * Fetches a 16-bit integer value from the given offset, swapping
    * bytes if needed.
    */
  def getInt16(buffer: Array[Byte], offset: Int, swap: Boolean): Short =
    if ( swap ) {
      ( (buffer(offset + 0) & 0xFF)       |
        (buffer(offset + 1) & 0xFF) <<  8    ).toShort
    } else {
      ( (buffer(offset + 0) & 0xFF) <<  8 |
        (buffer(offset + 1) & 0xFF)          ).toShort
    }

  /**
    * Fetches a 24-bit integer value from the given offset, swapping
    * bytes if needed.
    */
  def getInt24(buffer: Array[Byte], offset: Int, swap: Boolean): Int =
    if ( swap ) {
      (buffer(offset + 0) & 0xFF)       |
      (buffer(offset + 1) & 0xFF) <<  8 |
      (buffer(offset + 2) & 0xFF) << 16
    } else {
      (buffer(offset + 0) & 0xFF) << 16 |
      (buffer(offset + 1) & 0xFF) <<  8 |
      (buffer(offset + 2) & 0xFF)
    }

  /**
    * Fetches a 32-bit integer value from the given offset, swapping
    * bytes if needed.
    */
  def getInt32(buffer: Array[Byte], offset: Int, swap: Boolean): Int =
    if ( swap ) {
      (buffer(offset + 0) & 0xFF)       |
      (buffer(offset + 1) & 0xFF) <<  8 |
      (buffer(offset + 2) & 0xFF) << 16 |
      (buffer(offset + 3) & 0xFF) << 24
    } else {
      (buffer(offset + 0) & 0xFF) << 24 |
      (buffer(offset + 1) & 0xFF) << 16 |
      (buffer(offset + 2) & 0xFF) <<  8 |
      (buffer(offset + 3) & 0xFF)
    }

  /**
    * Fetches a 64-bit integer value from the given offset, swapping
    * bytes if needed.
    */
  def getInt64(buffer: Array[Byte], offset: Int, swap: Boolean): Long =
    if ( swap ) {
      (buffer(offset + 0).toLong & 0xFF)       |
      (buffer(offset + 1).toLong & 0xFF) <<  8 |
      (buffer(offset + 2).toLong & 0xFF) << 16 |
      (buffer(offset + 3).toLong & 0xFF) << 24 |
      (buffer(offset + 4).toLong & 0xFF) << 32 |
      (buffer(offset + 5).toLong & 0xFF) << 40 |
      (buffer(offset + 6).toLong & 0xFF) << 48 |
      (buffer(offset + 7).toLong & 0xFF) << 56
    } else {
      (buffer(offset + 0).toLong & 0xFF) << 56 |
      (buffer(offset + 1).toLong & 0xFF) << 48 |
      (buffer(offset + 2).toLong & 0xFF) << 40 |
      (buffer(offset + 3).toLong & 0xFF) << 32 |
      (buffer(offset + 4).toLong & 0xFF) << 24 |
      (buffer(offset + 5).toLong & 0xFF) << 16 |
      (buffer(offset + 6).toLong & 0xFF) <<  8 |
      (buffer(offset + 7).toLong & 0xFF)
    }

  /**
    * Fetches an array of bytes from the given offset.
    */
  def getBytes(buffer: Array[Byte], offset: Int, length: Int): Array[Byte] =
    buffer.slice(offset, offset + length)

  /**
    * Fetches a 64-bit start time expressed as milliseconds since the
    * UNIX epoch from the given offset, swapping bytes if needed, and
    * returns it as a [[java.time.Instant]].
    */
  def getStartTime(buffer: Array[Byte], offset: Int, swap: Boolean): Instant =
    Instant.ofEpochMilli(getInt64(buffer, offset, swap))

  /**
    * Fetches a 32-bit start time expressed as seconds since the UNIX
    * epoch from the given offset, swapping bytes if needed.
    */
  def getStartTimeSecs(buffer: Array[Byte], offset: Int, swap: Boolean): Instant =
    Instant.ofEpochSecond(
      JInt.toUnsignedLong(getInt32(buffer, offset, swap)))

  /**
    * Fetches a 32-bit duration expressed in milliseconds from the
    * given offset, swapping bytes if needed.
    */
  def getElapsed(buffer: Array[Byte], offset: Int, swap: Boolean): Duration =
    Duration.ofMillis(getInt32(buffer, offset, swap).toLong)

  /**
    * Fetches a 16-bit duration expressed in seconds from the
    * given offset, swapping bytes if needed.
    */
  def getElapsedSecs16(buffer: Array[Byte], offset: Int, swap: Boolean): Duration =
    Duration.ofSeconds(
      JShort.toUnsignedLong(getInt16(buffer, offset, swap)))

  /**
    * Fetches a 32-bit duration expressed in seconds from the given
    * offset, swapping bytes if needed.
    */
  def getElapsedSecs32(buffer: Array[Byte], offset: Int, swap: Boolean): Duration =
    Duration.ofSeconds(
      JInt.toUnsignedLong(getInt32(buffer, offset, swap)))

  /**
    * Fetches a 16-bit integer port value from the given offset,
    * swapping bytes if needed.
    */
  def getPort(buffer: Array[Byte], offset: Int, swap: Boolean): Port =
    Port(getInt16(buffer, offset, swap))

  /**
    * Fetches an 8-bit integer protocol value from the given offset.
    */
  def getProtocol(buffer: Array[Byte], offset: Int, swap: Boolean): Protocol =
    Protocol(getInt8(buffer, offset, swap))

  /**
    * Fetches an 8-bit integer SiLK flow type value from the given
    * offset.
    */
  def getFlowType(buffer: Array[Byte], offset: Int, swap: Boolean): FlowType =
    FlowType(getInt8(buffer, offset, swap))

  /**
    * Fetches an 8-bit integer SiLK sensor value from the given offset.
    */
  def getSensor8(buffer: Array[Byte], offset: Int, swap: Boolean): Sensor =
    Sensor(JByte.toUnsignedInt(getInt8(buffer, offset, swap)).toShort)

  /**
    * Fetches a 16-bit integer SiLK sensor value from the given
    * offset, swapping bytes if needed.
    */
  def getSensor16(buffer: Array[Byte], offset: Int, swap: Boolean): Sensor =
    Sensor(getInt16(buffer, offset, swap))

  /**
    * Fetches an 8-bit TCP flag value from the given offset.
    */
  def getTCPFlags(buffer: Array[Byte], offset: Int, swap: Boolean): TCPFlags =
    TCPFlags(getInt8(buffer, offset, swap))

  /**
    * Fetches an 8-bit SiLK TCP state value from the given offset.
    */
  def getTCPState(buffer: Array[Byte], offset: Int, swap: Boolean): TCPState =
    TCPState(getInt8(buffer, offset, swap))

  /**
    * Fetches a 16-bit port number representing application from the
    * given offset, swapping bytes if needed.
    */
  def getApplication(buffer: Array[Byte], offset: Int, swap: Boolean): Port =
    Port(getInt16(buffer, offset, swap))

  /**
    * Fetches a 16-bit integer representing a memo value from the
    * given offset, swapping bytes if needed.
    */
  def getMemo(buffer: Array[Byte], offset: Int, swap: Boolean): Short =
    getInt16(buffer, offset, swap)

  /**
    * Fetches an 8-bit integer representing an SNMP interface from the
    * given offset.
    */
  def getSNMPInterface8(
    buffer: Array[Byte], offset: Int, swap: Boolean): SNMPInterface =
      SNMPInterface(JByte.toUnsignedInt(getInt8(buffer, offset, swap)))

  /**
    * Fetches a 16-bit integer representing an SNMP interface from the
    * given offset, swapping bytes if needed.
    */
  def getSNMPInterface16(buffer: Array[Byte], offset: Int, swap: Boolean): SNMPInterface =
    SNMPInterface(JShort.toUnsignedInt(getInt16(buffer, offset, swap)))

  /**
    * Fetches a 32-bit integer representing an SNMP interface from the
    * given offset, swapping bytes if needed.
    */
  def getSNMPInterface32(buffer: Array[Byte], offset: Int, swap: Boolean): SNMPInterface =
    SNMPInterface(getInt32(buffer, offset, swap))

  /**
    * Fetches a 24-bit integer representing a packet count from the
    * given offset, swapping bytes if needed.
    */
  def getPackets24(buffer: Array[Byte], offset: Int, swap: Boolean): Long =
    getInt24(buffer, offset, swap).toLong

  /**
    * Fetches a 32-bit integer representing a packet count from the
    * given offset, swapping bytes if needed.
    */
  def getPackets32(buffer: Array[Byte], offset: Int, swap: Boolean): Long =
    JInt.toUnsignedLong(getInt32(buffer, offset, swap))

  /**
    * Fetches a 64-bit integer representing a packet count from the
    * given offset, swapping bytes if needed.
    */
  def getPackets64(buffer: Array[Byte], offset: Int, swap: Boolean): Long =
    getInt64(buffer, offset, swap)

  /**
    * Fetches a 32-bit integer representing a byte count from the
    * given offset, swapping bytes if needed.
    */
  def getBytes32(buffer: Array[Byte], offset: Int, swap: Boolean): Long =
    JInt.toUnsignedLong(getInt32(buffer, offset, swap))

  /**
    * Fetches a 64-bit integer representing a byte count from the
    * given offset, swapping bytes if needed.
    */
  def getBytes64(buffer: Array[Byte], offset: Int, swap: Boolean): Long =
    getInt64(buffer, offset, swap)

  /**
    * Fetches a 128-bit IPv6 address from the given offset (always in
    * network byte order.)
    */
  def getIPv6Address(buffer: Array[Byte], offset: Int): IPv6Address =
    IPv6Address(getBytes(buffer, offset, 16))

  /**
    * Fetches a 32-bit IPv4 address from the given offset, swapping
    * bytes if needed.
    */
  def getIPv4Address(buffer: Array[Byte], offset: Int, swap: Boolean = false): IPv4Address =
    IPv4Address(getInt32(buffer, offset, swap))

  // Decode the startTime, protocol, TCP state, and flags for formats
  // that encode these using rflag_stime, proto_iflags, and tcp_state:

  //  uint32_t      rflag_stime;     //  0- 3
  //  // uint32_t     rest_flags: 8; //        is_tcp==0: Empty; else
  //                                 //          EXPANDED==0:Empty
  //                                 //          EXPANDED==1:TCPflags/!1st pkt
  //  // uint32_t     is_tcp    : 1; //        1 if FLOW is TCP; 0 otherwise
  //  // uint32_t     unused    : 1; //        Reserved
  //  // uint32_t     stime     :22; //        Start time:msec offset from hour
  //
  //  uint8_t       proto_iflags;    //  4     is_tcp==0: Protocol; else:
  //                                 //          EXPANDED==0:TCPflags/ALL pkts
  //                                 //          EXPANDED==1:TCPflags/1st pkt
  //  uint8_t       tcp_state;       //  5     TCP state machine info

  /**
    * Decodes bit-bashed information providing the start time,
    * protocol, TCP state, and flags for SiLK record formats that use
    * this representation.
    */
  def unpackTimesFlagsProto(
    buffer: Array[Byte], offset: Int, swap: Boolean, header: Header):
      (Instant, Protocol, TCPState, TCPFlags, TCPFlags, TCPFlags) =
  {
    val rflagStartTime = getInt32(buffer, offset, swap)
    val tcpState = getTCPState(buffer, offset + 5, swap)
    val startTimeOffset = rflagStartTime & 0x003FFFFF
    val isTCP = (rflagStartTime & 0x00800000) != 0

    val startTime = Instant.ofEpochMilli(
      header.packedStartTime + startTimeOffset)
    if ( !isTCP ) {
      (startTime, getProtocol(buffer, offset + 4, false), tcpState,
        TCPFlags(0), TCPFlags(0), TCPFlags(0))
    } else if ( tcpState.expandedFlags ) {
      val restFlags = (rflagStartTime >> 24).toByte
      val initFlags = getInt8(buffer, offset + 4, false)
      val flags = (restFlags | initFlags).toByte
      (startTime, Protocol.TCP, tcpState,
        TCPFlags(flags), TCPFlags(initFlags), TCPFlags(restFlags))
    } else {
      (startTime, Protocol.TCP, tcpState,
        getTCPFlags(buffer, offset + 4, false), TCPFlags(0), TCPFlags(0))
    }
  }

  // Decode the sTime, elapsed, packets, bytes, protocol, TCP flags,
  // TCP state, and application fields for formats that encode these
  // using stime_bb1, bb2_elapsed, pro_flg_pkts, tcp_state, rest_flags,
  // and application:

  // in the following: EXPANDED == ((tcp_state & SK_TCPSTATE_EXPANDED) ? 1 : 0)
  //
  //  uint32_t      stime_bb1;       //  0- 3
  //  // uint32_t     stime     :22  //        Start time:msec offset from hour
  //  // uint32_t     bPPkt1    :10; //        Whole bytes-per-packet (hi 10)
  //
  //  uint32_t      bb2_elapsed;     //  4- 7
  //  // uint32_t     bPPkt2    : 4; //        Whole bytes-per-packet (low 4)
  //  // uint32_t     bPPFrac   : 6; //        Fractional bytes-per-packet
  //  // uint32_t     elapsed   :22; //        Duration of flow in msec
  //
  //  uint32_t      pro_flg_pkts;    //  8-11
  //  // uint32_t     prot_flags: 8; //        is_tcp==0: IP protocol
  //                                 //        is_tcp==1 &&
  //                                 //          EXPANDED==0:TCPflags/All pkts
  //                                 //          EXPANDED==1:TCPflags/1st pkt
  //  // uint32_t     pflag     : 1; //        'pkts' requires multiplier?
  //  // uint32_t     is_tcp    : 1; //        1 if flow is TCP; 0 otherwise
  //  // uint32_t     padding   : 2; //
  //  // uint32_t     pkts      :20; //        Count of packets
  //
  //  uint8_t       tcp_state;       // 12     TCP state machine info
  //  uint8_t       rest_flags;      // 13     is_tcp==0: Flow's reported flags
  //                                 //        is_tcp==1 &&
  //                                 //          EXPANDED==0:Empty
  //                                 //          EXPANDED==1:TCPflags/!1st pkt
  //  uint16_t      application;     // 14-15  Type of traffic

  /**
    * Decodes bit-bashed information providing the start time,
    * duration, packet count, byte count, protocol, TCP flags, TCP
    * state, and application fields for SiLK record formats that use
    * this representation.
    */
  def unpackFlagsTimesVolumes(
    buffer: Array[Byte], offset: Int, len: Int, inIsTCP: Boolean,
    swap: Boolean, header: Header):
      (Instant, Duration, Long, Long, Protocol,
        TCPFlags, TCPFlags, TCPFlags, TCPState, Port) =
  {
    val (tcpState, inRestFlags, application) = {
      if ( len == 12 ) {
        (TCPState(0), TCPFlags(0), Port(0))
      } else {
        (getTCPState(buffer, offset + 12, swap),
          getTCPFlags(buffer, offset + 13, swap),
          getApplication(buffer, offset + 14, swap))
      }
    }
    val protoFlagPackets = getInt32(buffer, offset + 8, swap)
    val inPackets = protoFlagPackets & 0x000FFFFF
    val pFlag = (protoFlagPackets & 0x00800000) != 0
    val isTCP = if ( inIsTCP ) {
      true
    } else {
      (protoFlagPackets & 0x00400000) != 0
    }
    val (protocol, flags, initFlags, restFlags) =
      if ( !isTCP ) {
        (Protocol((protoFlagPackets >> 24).toByte),
          inRestFlags, TCPFlags(0), TCPFlags(0))
      } else {
        val initFlags = TCPFlags((protoFlagPackets >> 24).toByte)
        val flags = TCPFlags((initFlags.toByte | inRestFlags.toByte).toByte)
        if ( tcpState.expandedFlags ) {
          (Protocol.TCP, flags, initFlags, inRestFlags)
        } else {
          (Protocol.TCP, flags, TCPFlags(0), TCPFlags(0))
        }
      }
    val bb2Elapsed = getInt32(buffer, offset + 4, swap)
    val elapsed = Duration.ofMillis((bb2Elapsed & 0x003FFFFF).toLong)
    val stimeBb1 = getInt32(buffer, offset, swap)
    val startTimeOffset = (stimeBb1 >> 10) & 0x003FFFFF
    val startTime = Instant.ofEpochMilli(
      header.packedStartTime + startTimeOffset)
    val bpp =
      ((stimeBb1 & 0x000003FF) << 10) | ((bb2Elapsed >> 22) & 0x000003FF)
    val (bytes, packets) = unpackBytesPackets(bpp, inPackets, pFlag)
    (startTime, elapsed, packets, bytes, protocol,
      flags, initFlags, restFlags, tcpState, application)
  }

  // Decode the startTime, packets, bytes, elapsed, protocol, and
  // flags fields for formats that encode these using pkts_stime, bbe,
  // and msec_flags:

  //  uint32_t      pkts_stime
  //  // uint32_t     pkts      :20; //        Count of packets
  //  // uint32_t     sTime     :12; //        Start time--offset from hour
  //
  //  uint32_t      bbe
  //  // uint32_t     bPPkt     :14; //        Whole bytes-per-packet
  //  // uint32_t     bPPFrac   : 6; //        Fractional bytes-per-packet
  //  // uint32_t     elapsed   :12; //        Duration of flow
  //
  //  uint32_t      msec_flags
  //  // uint32_t     sTime_msec:10; //        Fractional sTime (millisec)
  //  // uint32_t     elaps_msec:10; //        Fractional elapsed (millisec)
  //  // uint32_t     pflag     : 1; //        'pkts' requires multiplier?
  //  // uint32_t     is_tcp    : 1; //        1 if flow is TCP; 0 otherwise
  //  // uint32_t     padding   : 2; //        padding/reserved
  //  // uint32_t     prot_flags: 8; //        is_tcp==0: IP protocol
  //                                 //        is_tcp==1 &&
  //                                 //          EXPANDED==0:TCPflags/All pkts
  //                                 //          EXPANDED==1:TCPflags/1st pkt

  /**
    * Decodes bit-bashed information providing the start time, packet
    * count, byte count, duration, protocol, and TCP flags fields for
    * SiLK record formats that use this representation.
    */
  def getTimeBytesPacketsFlags(
    buffer: Array[Byte], offset1: Int, offset2: Int, offset3: Int,
    forceTCP: Boolean, swap: Boolean, header: Header):
      (Instant, Long, Long, Duration, Protocol, TCPFlags) =
  {
    val pktsStime = getInt32(buffer, offset1, swap)
    val bbe = getInt32(buffer, offset2, swap)
    val msecFlags = getInt32(buffer, offset3, swap)

    val inPackets = (pktsStime >> 12) & 0x000FFFFF

    val startTimeSecOffset = pktsStime & 0x00000FFF
    val startTimeMilliOffset = (msecFlags >> 22) & 0x000003FF
    val startTime = Instant.ofEpochMilli(
      header.packedStartTime + startTimeSecOffset * 1000 +
        startTimeMilliOffset)

    val bytesPerPacket = (bbe >> 12) & 0x000FFFFF

    val elapsedSecs = bbe & 0x00000FFF
    val elapsedMillis = (msecFlags >> 12) & 0x000003FF
    val elapsed = Duration.ofMillis(
      (elapsedSecs * 1000 + elapsedMillis).toLong)

    val pflag = (msecFlags & 0x00000800) != 0
    val isTCP = ((msecFlags & 0x00000400) != 0) || forceTCP

    val (bytes, packets) = unpackBytesPackets(bytesPerPacket, inPackets, pflag)

    val protFlags = msecFlags.toByte

    if ( isTCP ) {
      (startTime, packets, bytes, elapsed, Protocol.TCP, TCPFlags(protFlags))
    } else {
      (startTime, packets, bytes, elapsed, Protocol(protFlags), TCPFlags(0))
    }
  }

  // Convert fixed-point bytes-per-packet and packets values to bytes
  // and packet values for formats requiring this
  def unpackBytesPackets(
    bpp: Int, inPackets: Int, pflag: Boolean): (Long, Long) =
  {
    val packets = if ( pflag ) { inPackets * 64 } else { inPackets }
    val bytesPerPacket = (bpp >> 6) & 0x000003FFF
    val bytesPerPacketFrac = bpp & 0x0000003F
    val i_quot = (bytesPerPacketFrac * packets) >> 6
    val i_rem = (bytesPerPacketFrac * packets) & 0x3F
    val bytes = ( (bytesPerPacket * packets) + i_quot +
                  (if ( i_rem >= 32 ) 1 else 0) )
    (bytes.toLong, packets.toLong)
  }

  // Figure out the protocol and TCP flags fields for formats that share
  // the same bytes for protocol/flags with a bit for whether it's TCP

  /**
    * Works out what protocol and flag information is in use for some
    * of the bit-bashed formats.
    */
  def unpackProtoFlags(
    isTCP: Boolean, protFlags: Byte, tcpState: TCPState, inRestFlags: Byte):
      (Protocol, TCPFlags, TCPFlags, TCPFlags) =
  {
    if ( isTCP ) {
      if ( tcpState.expandedFlags ) {
        val flags = TCPFlags((protFlags | inRestFlags).toByte)
        val initFlags = TCPFlags(protFlags)
        val restFlags = TCPFlags(inRestFlags)
        (Protocol.TCP, flags, initFlags, restFlags)
      } else {
        (Protocol.TCP, TCPFlags(protFlags), TCPFlags(0), TCPFlags(0))
      }
    } else {
      (Protocol(protFlags), TCPFlags(inRestFlags), TCPFlags(0), TCPFlags(0))
    }
  }

  /**
    * Works out the start time, duration, byte count, and packet count
    * for some of the bit-bashed formats.
    */
  def unpackSbbPef(
    buffer: Array[Byte], sbbOffset: Int, pefOffset: Int, swap: Boolean,
    header: Header): (Instant, Duration, Long, Long) =
  {
    val pef = getInt32(buffer, pefOffset, swap)
    val inPackets = (pef >> 12) & 0x000FFFFF
    val elapsedSecs = (pef >> 1) & 0x000007FF
    val pFlag = (pef & 0x00000001) != 0
    val sbb = getInt32(buffer, sbbOffset, swap)
    val bpp = sbb & 0x000FFFFF
    val startTimeSecOffset = (sbb >> 20) & 0x00000FFF
    val startTime = Instant.ofEpochMilli(
      header.packedStartTime + startTimeSecOffset * 1000)
    val elapsed = Duration.ofSeconds(elapsedSecs.toLong)
    val (bytes, packets) = unpackBytesPackets(bpp, inPackets, pFlag)
    (startTime, elapsed, bytes, packets)
  }

  /**
    * Decodes the bit-bashed "web port" information for SiLK web
    * record formats that use a condensed form for that information.
    */
  def decodeWWWPort(bits: Int): Port = bits match {
    case 0 => Port(80)
    case 1 => Port(443)
    case 2 => Port(8080)
    case _ => Port(0)
  }

  /**
    * Provides a helper method to RWRec objects for some formats which
    * need a sanity check to see if initFlags and restFlags are set
    * despite the protocol not being TCP or the expandedFlags bit not
    * being set. This runs that check and generates a new RWRec with
    * the fix if needed.
    */
  implicit class TCPStateFixHelper(val self: RWRec) {
    def maybeClearTCPStateExpanded: RWRec = {
      if ( self.tcpState.expandedFlags &&
        (self.protocol != Protocol.TCP ||
          (self.initFlags.toByte == 0 && self.restFlags.toByte == 0)) ) {
        RWRec(self.startTime, self.elapsed, self.sPort, self.dPort,
          self.protocol, self.flowType, self.sensor, self.flags,
          TCPFlags(0), TCPFlags(0),
          TCPState((self.tcpState.toByte & ~0x01).toByte),
          self.application, self.memo, self.input, self.output,
          self.packets, self.bytes, self.sIP, self.dIP, self.nhIP)
      } else {
        self
      }
    }
  }




  /**
    * Updates 1 byte at the given offset with an 8-bit integer value.
    */
  def putInt8(buffer: Array[Byte], offset: Int, value: Byte): Unit =
    buffer(offset + 0) = value

  /**
    * Updates 2 bytes starting at the given offset with a 16-bit
    * integer value written in network byte order.
    */
  def putInt16(buffer: Array[Byte], offset: Int, value: Short): Unit = {
    buffer(offset + 0) = (value >>> 8).toByte
    buffer(offset + 1) = (value      ).toByte
  }

  /**
    * Updates 4 bytes starting at the given offset with a 32-bit
    * integer value written in network byte order.
    */
  def putInt32(buffer: Array[Byte], offset: Int, value: Int): Unit = {
    buffer(offset + 0) = (value >>> 24).toByte
    buffer(offset + 1) = (value >>> 16).toByte
    buffer(offset + 2) = (value >>>  8).toByte
    buffer(offset + 3) = (value       ).toByte
  }

  /**
    * Updates 8 bytes starting at the given offset with a 64-bit
    * integer value written in network byte order.
    */
  def putInt64(buffer: Array[Byte], offset: Int, value: Long): Unit = {
    buffer(offset + 0) = (value >>> 56).toByte
    buffer(offset + 1) = (value >>> 48).toByte
    buffer(offset + 2) = (value >>> 40).toByte
    buffer(offset + 3) = (value >>> 32).toByte
    buffer(offset + 4) = (value >>> 24).toByte
    buffer(offset + 5) = (value >>> 16).toByte
    buffer(offset + 6) = (value >>>  8).toByte
    buffer(offset + 7) = (value       ).toByte
  }

  /**
    * Updates bytes starting at the given offset with all the bytes in
    * another array.
    */
  def putBytes(buffer: Array[Byte], offset: Int, value: Array[Byte]): Unit = {
    // copyToArray stops copying when it runs into the end of either
    // the input or the output, let's double-check
    assert(value.length + offset <= buffer.length)
    value.copyToArray(buffer, offset, value.length)
    ()
  }

  /**
    * Updates bytes starting at the given offset with no more then
    * `length` bytes taken from another array (stating at index 0 of
    * that array).
    */
  def putBytes(
    buffer: Array[Byte], offset: Int, value: Array[Byte], length: Int
  ): Unit = {
    // copyToArray stops copying when it runs into the end of either
    // the input or the output, let's double-check
    assert(length <= value.length)
    assert(length + offset <= buffer.length)
    value.copyToArray(buffer, offset, length)
    ()
  }

  /**
    * Updates 8 bytes starting at the given offset with a 64-bit
    * integer value, written in network byte order, representing the
    * number of epoch milliseconds in the time `value`.
    */
  def putInstant(buffer: Array[Byte], offset: Int, value: Instant): Unit =
    putInt64(buffer, offset, value.toEpochMilli())

  /**
    * Updates 4 bytes starting at the given offset with a 32-bit
    * integer value, written in network byte order, representing a
    * duration expressed in milliseconds.
    */
  def putElapsed(buffer: Array[Byte], offset: Int, value: Duration): Unit =
    putInt32(buffer, offset, value.toMillis().toInt)

  /**
    * Updates 2 bytes starting at the given offset with a 16-bit
    * integer value, written in network byte order, representing a
    * port number.
    */
  def putPort(buffer: Array[Byte], offset: Int, value: Port): Unit =
    putInt16(buffer, offset, value.toShort)

  /**
    * Updates 1 byte at the given offset with an 8-bit integer
    * protocol value.
    */
  def putProtocol(buffer: Array[Byte], offset: Int, value: Protocol): Unit =
    putInt8(buffer, offset, value.toByte)

  /**
    * Updates 1 byte at the given offset with an 8-bit integer SiLK
    * flow type value.
    */
  def putFlowType(buffer: Array[Byte], offset: Int, value: FlowType): Unit =
    putInt8(buffer, offset, value.toByte)

  /**
    * Updates 2 bytes starting at the given offset with a 16-bit
    * integer SiLK sensor value written in network byte order.
    */
  def putSensor(buffer: Array[Byte], offset: Int, value: Sensor): Unit =
    putInt16(buffer, offset, value.toShort)

  /**
    * Updates 1 byte at the given offset with an 8-bit TCP flag value.
    */
  def putTCPFlags(buffer: Array[Byte], offset: Int, value: TCPFlags): Unit =
    putInt8(buffer, offset, value.toByte)

  /**
    * Updates 1 byte at the given offset with an 8-bit SiLK TCP state
    * (attribute) value.
    */
  def putTCPState(buffer: Array[Byte], offset: Int, value: TCPState): Unit =
    putInt8(buffer, offset, value.toByte)

  /**
    * Updates 2 bytes starting at the given offset with a 16-bit
    * integer port value, written in network byte order, representing
    * an application.
    */
  def putApplication(buffer: Array[Byte], offset: Int, value: Port): Unit =
    putInt16(buffer, offset, value.toShort)

  /**
    * Updates 2 bytes starting at the given offset with a 16-bit
    * integer value, written in network byte order, representing a
    * memo value.
    */
  def putMemo(buffer: Array[Byte], offset: Int, value: Short): Unit =
    putInt16(buffer, offset, value)

  /**
    * Updates 2 bytes starting at the given offset with a 16-bit
    * integer value, written in network byte order, representing the
    * lower two bytes of an SNMP interface value.
    */
  def putSNMPInterface16(buffer: Array[Byte], offset: Int, value: SNMPInterface): Unit =
    putInt16(buffer, offset, value.toShort)

  /**
    * Updates 4 bytes starting at the given offset with a 32-bit
    * integer value, written in network byte order, representing an
    * SNMP interface value.
    */
  def putSNMPInterface32(buffer: Array[Byte], offset: Int, value: SNMPInterface): Unit =
    putInt32(buffer, offset, value.toInt)

  /**
    * Updates 4 bytes starting at the given offset with a 32-bit
    * integer value, written in network byte order, representing a
    * packet count.
    */
  def putPackets32(buffer: Array[Byte], offset: Int, value: Long): Unit =
    putInt32(buffer, offset, value.toInt)

  /**
    * Updates 8 bytes starting at the given offset with a 64-bit
    * integer value, written in network byte order, representing a
    * packet count.
    */
  def putPackets64(buffer: Array[Byte], offset: Int, value: Long): Unit =
    putInt64(buffer, offset, value)

  /**
    * Updates 4 bytes starting at the given offset with a 32-bit
    * integer value, written in network byte order, representing a
    * byte count.
    */
  def putBytes32(buffer: Array[Byte], offset: Int, value: Long): Unit =
    putInt32(buffer, offset, value.toInt)

  /**
    * Updates 8 bytes starting at the given offset with a 64-bit
    * integer value, written in network byte order, representing a
    * byte count.
    */
  def putBytes64(buffer: Array[Byte], offset: Int, value: Long): Unit =
    putInt64(buffer, offset, value)

  /**
    * How to fill in the first 12 bytes of a 16 byte value when
    * encoding an IPv4 address in an array that expects IPv6
    * addresses.
    */
  private[this] val ipv4InV6: Array[Byte] = Array(0, 0, 0, 0, 0, 0, 0, 0, 0, 0, -1, -1)

  /**
    * Updates 16 bytes starting at the given offset with an IP
    * address.  When `value` is an IPv6 address, the address is copied
    * to the bytes.  When `value` in an IPv4 address, an IPv6 encoding
    * of the IPv4 address is copied.
    *
    * Returns `true` if `value` is IPv6 and `false` if it is IPv4.
    */
  def putIPAddress(buffer: Array[Byte], offset: Int, value: IPAddress): Boolean = {
    value match {
      case ipv6: IPv6Address => {
        putBytes(buffer, offset, ipv6.toBytes, 16)
        true
      }
      case ipv4: IPv4Address => {
        putBytes(buffer, offset, ipv4InV6, 12)
        putInt32(buffer, offset + 12, ipv4.toInt)
        false
      }
    }
  }

  /**
    * Updates 16 bytes starting at the given offset with a 128-bit
    * IPv6 address.
    */
  def putIPv6Address(buffer: Array[Byte], offset: Int, value: IPv6Address): Unit =
    putBytes(buffer, offset, value.toBytes, 16)

  /**
    * Updates 4 bytes starting at the given offset with a 32-bit IPv4
    * address written in network byte order.
    */
  def putIPv4Address(buffer: Array[Byte], offset: Int, value: IPv4Address): Unit =
    putInt32(buffer, offset, value.toInt)

}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
