// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.silk
package io

import java.io.{DataOutputStream, OutputStream}

import org.anarres.lzo.{
  LzoAlgorithm, LzoConstraint, LzoCompressor, LzoLibrary,
  LzoTransformer, lzo_uintp}

/**
  * A [[BufferWriter]] which compresses data using LZO and writes it
  * to a stream in a form compatible with SiLK file streams.
  *
  * @see [[LzoOutputStreamBuffer$ the companion object]] for more
  * details.
  */
private[silk] class LzoOutputStreamBuffer(outStream: OutputStream)
    extends BufferWriter
{
  // The output stream
  private val out: DataOutputStream = outStream match {
    case out: DataOutputStream => out
    case _ => new DataOutputStream(outStream)
  }

  private val compressor: LzoCompressor =
    LzoLibrary.getInstance().newCompressor(
      LzoAlgorithm.LZO1X, LzoConstraint.COMPRESSION)

  /** The size of a buffer that is guaranteed to be large enough to hold
    * the result of compressing `sourceLen` bytes of data.
    */
  private def compressBound(sourceLen: Int): Int = {
    // Formula taken from the LZO FAQ:
    // http://www.oberhumer.com/opensource/lzo/lzofaq.php
    sourceLen + (sourceLen >>> 4) + 64 + 3
  }

  /** Finish writing compressed data. In this case, flush data written
    * to the output stream. */
  def end(): Unit = {
    out.flush()
  }

  /** Write the first `length` bytes of data in `buffer` to the output
    * stream.
    *
    * @throws java.io.IOException when an error occurs.
    */
  def putBuffer(uncompressedData: Array[Byte], uncompressedSize: Int): Unit =
    if ( uncompressedSize > 0 ) {

      // create buffer to hold the result.  FIXME: Consider moving this
      // array outside the function
      val maxCompressedSize = compressBound(uncompressedSize)
      val compressedData = Array.ofDim[Byte](maxCompressedSize)

      val outLen = new lzo_uintp(maxCompressedSize)
      // compress it
      compressor.compress(
        uncompressedData, 0, uncompressedSize,
        compressedData, 0, outLen) match
      {
        case LzoTransformer.LZO_E_OK =>
          out.writeInt(outLen.value)
          out.writeInt(uncompressedSize)
          out.write(compressedData, 0, outLen.value)
        case rc =>
          sys.error(
            s"Failure compressing: ${compressor.toErrorString(rc)}")
      }
    }
}

/** Support for creating an [[LzoOutputStreamBuffer]]. */
private[silk] object LzoOutputStreamBuffer {
  def apply(outStream: OutputStream): LzoOutputStreamBuffer =
    new LzoOutputStreamBuffer(outStream)
}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
