// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.silk
package io
package ipset

import BufferUtil.{
  getInt8, getInt64}

import org.cert.netsa.data.net.{
  IPv6Address, IPBlock, IPv6Block}


/*
 *    IPSET_REC_VERSION_RADIX
 *    =======================
 *
 *    The IPset file format introduced in SiLK-3.0.  Files in this
 *    format may store either IPv4 or IPv6 addresses (a header entry
 *    specifies which).
 *
 *    This version contains a dump of the radix-tree data structure
 *    that is used in-memory.  The reasoning behind this was that the
 *    data could be mmap()ed directly from disk without having to read
 *    the file into memory.  For this to work, the file's data must
 *    not be compressed, the byte-order of the file must match the
 *    machine's byte-order, and the stream must be seekable (not
 *    standard input).
 *
 *    The radix data structure maintains the (interior) nodes separate
 *    from the leaves.  In the output stream, all nodes are written
 *    first, then the leaves. The leaves are sorted by IP before being
 *    written, and the first leaf in the output stream does not
 *    contain data.  An individual leaf is 8 octets for IPv4 and 24
 *    octets for IPv6.  For IPv4, a leaf contains the CIDR prefix in
 *    the first octet, three empty octets, and the 32-bit IPv4 address
 *    in native byte order.  For IPv6, a leaf contains the CIDR prefix
 *    in the first octet, seven(!) empty octets, followed by two
 *    unsigned 64-bit numbers is native byte order.  The first 64-bit
 *    number contains the upper 8 octets of the IPv6 address, and the
 *    second number contains the lower 8 octets.
 *
 *    The file has a header entry that may be used to determine
 *    whether the file contains IPv4 or IPv6 addresses.  The contents
 *    of the header are the number of children per node, the number of
 *    leaves, the size of a leaf, the number of (interior) nodes, the
 *    size of a node, and the location of the root node.  The leaf
 *    size is 8 for IPv4 and 24 for IPv6.
 *
 *    When reading from disk, skip over the nodes and read the data in
 *    the leaves.
 */


private[silk] class IPSetV3IPv6Reader(
  reader: BufferReader,
  header: Header
) extends IPSetReader(reader, header)
{
  // get the IPset header entry
  val hentry_opt = header.headerEntries.collectFirst({
    case entry : HeaderEntry.IPSet => entry
  })
  val hentry = hentry_opt.get

  private[this] val recordLength = hentry.leafSize

  //  As part of initialization: read and skip the nodes that appear
  //  in the IPSet file, and then read and skip the first leaf which
  //  contains no usable value.
  skipBytes(hentry.nodeSize * hentry.nodeCount + hentry.leafSize)

  override def containsIPv6: Boolean = true

  override def hasNext: Boolean = {
    checkAvailable(recordLength)
  }

  override def next(): IPBlock = {
    if ( !checkAvailable(recordLength) ) {
      throw new NoSuchElementException("IPSetV3IPv6Reader")
    }
    val cidr = (0xffL & getInt8(buffer, bufOffset, swap)).toInt
    val upper64 = getInt64(buffer, bufOffset + 8, swap)
    val lower64 = getInt64(buffer, bufOffset + 16, swap)
    bufOffset = bufOffset + recordLength
    val shorts: Array[Short] = Array(
      ((upper64 >>> 48) & 0xffff).toShort,
      ((upper64 >>> 32) & 0xffff).toShort,
      ((upper64 >>> 16) & 0xffff).toShort,
      ( upper64         & 0xffff).toShort,
      ((lower64 >>> 48) & 0xffff).toShort,
      ((lower64 >>> 32) & 0xffff).toShort,
      ((lower64 >>> 16) & 0xffff).toShort,
      ( lower64         & 0xffff).toShort)
    IPv6Block(IPv6Address(shorts), cidr)
  }
}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
