// Copyright 2015-2022 by Carnegie Mellon University
// See license information in LICENSE.txt

package org.cert.netsa.io.silk
package io
package unpackers

import BufferUtil._

/*
**  RWGENERIC VERSION 5
**
**    int64_t       sTime;           //  0- 7  Flow start time as milliseconds
**                                   //        since UNIX epoch
**
**    uint32_t      elapsed;         //  8-11  Duration of flow in milliseconds
**                                   //        (Allows for a 49 day flow)
**
**    uint16_t      sPort;           // 12-13  Source port
**    uint16_t      dPort;           // 14-15  Destination port
**
**    uint8_t       proto;           // 16     IP protocol
**    uint8_t       flow_type;       // 17     Class & Type info
**    uint16_t      sID;             // 18-19  Sensor ID
**
**    uint8_t       flags;           // 20     OR of all flags (Netflow flags)
**    uint8_t       init_flags;      // 21     TCP flags in first packet
**                                   //        or blank for "legacy" data
**    uint8_t       rest_flags;      // 22     TCP flags on non-initial packet
**                                   //        or blank for "legacy" data
**    uint8_t       tcp_state;       // 23     TCP state machine info (below)
**
**    uint16_t      application;     // 24-25  Indication of type of traffic
**    uint16_t      memo;            // 26-27  Application specific field
**
**    uint16_t      input;           // 28-29  Router incoming SNMP interface
**    uint16_t      output;          // 30-31  Router outgoing SNMP interface
**
**    uint32_t      pkts;            // 32-35  Count of packets
**    uint32_t      bytes;           // 36-39  Count of bytes
**
**    uint32_t      sIP;             // 40-43  Source IP
**    uint32_t      dIP;             // 44-47  Destination IP
**    uint32_t      nhIP;            // 48-51  Router Next Hop IP
**
**
**  52 bytes on disk.
*/

private[silk] object FT_RWGENERIC_v5 extends Unpacker {
  val recordLength = 52
  def unpack(
    buffer: Array[Byte], offset: Int, swap: Boolean, header: Header): RWRec =
  {
    val startTime = getStartTime(buffer, offset, swap)
    val elapsed = getElapsed(buffer, offset + 8, swap)
    val sPort = getPort(buffer, offset + 12, swap)
    val dPort = getPort(buffer, offset + 14, swap)
    val protocol = getProtocol(buffer, offset + 16, swap)
    val flowType = getFlowType(buffer, offset + 17, swap)
    val sensor = getSensor16(buffer, offset + 18, swap)
    val flags = getTCPFlags(buffer, offset + 20, swap)
    val initFlags = getTCPFlags(buffer, offset + 21, swap)
    val restFlags = getTCPFlags(buffer, offset + 22, swap)
    val tcpState = getTCPState(buffer, offset + 23, swap)
    val application = getPort(buffer, offset + 24, swap)
    val memo = getInt16(buffer, offset + 26, swap)
    val input = getSNMPInterface16(buffer, offset + 28, swap)
    val output = getSNMPInterface16(buffer, offset + 30, swap)
    val packets = getPackets32(buffer, offset + 32, swap)
    val bytes = getBytes32(buffer, offset + 36, swap)
    val sIP = getIPv4Address(buffer, offset + 40, swap)
    val dIP = getIPv4Address(buffer, offset + 44, swap)
    val nhIP = getIPv4Address(buffer, offset + 48, swap)
    RWRec(
      startTime, elapsed, sPort, dPort, protocol, flowType, sensor,
      flags, initFlags, restFlags, tcpState, application, memo, input,
      output, packets, bytes, sIP, dIP, nhIP).maybeClearTCPStateExpanded
  }
}

// @LICENSE_FOOTER@
//
// Copyright 2015-2022 Carnegie Mellon University. All Rights Reserved.
//
// This material is based upon work funded and supported by the
// Department of Defense and Department of Homeland Security under
// Contract No. FA8702-15-D-0002 with Carnegie Mellon University for the
// operation of the Software Engineering Institute, a federally funded
// research and development center sponsored by the United States
// Department of Defense. The U.S. Government has license rights in this
// software pursuant to DFARS 252.227.7014.
//
// NO WARRANTY. THIS CARNEGIE MELLON UNIVERSITY AND SOFTWARE ENGINEERING
// INSTITUTE MATERIAL IS FURNISHED ON AN "AS-IS" BASIS. CARNEGIE MELLON
// UNIVERSITY MAKES NO WARRANTIES OF ANY KIND, EITHER EXPRESSED OR
// IMPLIED, AS TO ANY MATTER INCLUDING, BUT NOT LIMITED TO, WARRANTY OF
// FITNESS FOR PURPOSE OR MERCHANTABILITY, EXCLUSIVITY, OR RESULTS
// OBTAINED FROM USE OF THE MATERIAL. CARNEGIE MELLON UNIVERSITY DOES NOT
// MAKE ANY WARRANTY OF ANY KIND WITH RESPECT TO FREEDOM FROM PATENT,
// TRADEMARK, OR COPYRIGHT INFRINGEMENT.
//
// Released under a GNU GPL 2.0-style license, please see LICENSE.txt or
// contact permission@sei.cmu.edu for full terms.
//
// [DISTRIBUTION STATEMENT A] This material has been approved for public
// release and unlimited distribution. Please see Copyright notice for
// non-US Government use and distribution.
//
// Carnegie Mellon(R) and CERT(R) are registered in the U.S. Patent and
// Trademark Office by Carnegie Mellon University.
//
// This software includes and/or makes use of third party software each
// subject to its own license as detailed in LICENSE-thirdparty.tx
//
// DM20-1143
